/**
 * The view for a Confluence 'Page'. This view is analogous to the 'viewpage' of the desktop app which is to
 * say it represents more than just the content of a page or blog. This view encapsulates Content as well as
 * related concepts such as Comments, page level Likes and CommentLikes.
 */
ConfluenceMobile.PageView = Backbone.View.extend({

    el: ".container",

    events: {
        "click a" : "contentLink"
    },

    initialize: function (options) {
        options = options || {};

        if (!options.contentId) {
            throw new Error("contentId is a required parameter");
        }

        this.validPage = options.contentId.match(/(\d+)/)
            && options.contentId > 0;

        this.$el.empty();

        this.notification = new ConfluenceMobile.Notification(this.el);

        if (!this.validPage) {
            this.notification.showMessage(AJS.I18n.getText("confluence.mobile.profile.pageid.invalid"));
            return;
        }

        this.contentId = options.contentId;

        this.commentId = options.commentId || null;

        _.bindAll(this); // ensures that all references to "this" in the body of this view's functions actually refer to "this" instance.

        /**
         * Render to temporary container to avoid page reflow and produce smoother rendering. Once render to temp container is complete, append to DOM.
         */
        this.$temp = $("<div/>");

        var content = new ConfluenceMobile.Content({ id: this.contentId });

        content.bind("error", this.errorHandler);

        content.fetch({
            success: this.successHandler,
            timeout: 10e3
        });
    },

    /**
     * The PageView is a "super view" bringing together various sub views. So rendering simply involves
     * indicating 'busyness'. The subviews will render as triggered by events.
     */
    render: function() {
        this.validPage && this.notification.showLoading();
    },

    successHandler: function (model, response) {
        this.$temp.append(new ConfluenceMobile.ContentView({
            model: model
        }).render().el);

        ConfluenceMobile.contentEventAggregator.trigger("render:pre:after-content", this, this.$temp, model);

        var commentList = new ConfluenceMobile.CommentList(response.comments, {
            contentId: this.contentId,
            contextPath: ConfluenceMobile.AppData.get("confluence-context-path")
        });

        if (response.space && _.contains(response.space.permissions || [], "comment")) {
            commentList.setCanComment(true);
        }

        this.$temp.append(new ConfluenceMobile.CommentListView({
            model: commentList
        }).render().$el.wrap("<div class='comments'/>").parent());

        response.contentType && ConfluenceMobile.changeView("view-" + response.contentType);

        this.notification.hide();
        var that = this;

        // render is complete but only to the $temp node. Not yet rendered to the page.
        ConfluenceMobile.pageEventAggregator.trigger("post-render");

        setTimeout(function() {
            that.$el.html(that.$temp.unwrap()).show();

            // The version of Zepto used doesn't support a function passed to `filter()`.
            // Don't `imageViewer()` any images that are links.
            $(_.filter($(".confluence-embedded-image"), function (img) {
                return !$(img).closest("a").length;
            })).imageViewer();

            var boundHighlightLinkedComment = _.bind(that.highlightLinkedComment, that);

            /**
             * This invocation is responsible for highlighting the linked comment when the comment list is first rendered
             */
            setTimeout(boundHighlightLinkedComment, 0);

            /**
             * This is to handle the case where the user clicks on the date of another comment "on the same page".
             */
            window.addEventListener("hashchange", boundHighlightLinkedComment);

            // The page has been rendered and displayed
            ConfluenceMobile.contentEventAggregator.trigger("displayed");
        }, 0);

    },

    highlightLinkedComment: function () {

        var commentId = window.location.hash.split("/").pop().trim(),
            linkedClass = "linked";

        // Make sure a hash was passed and that it validates the
        // format for a comment link.
        if (!/^\d+$/.test(commentId)) {
            return;
        }

        var $comment = this.$("#comment-" + commentId);

        if (!$comment.length) {
            return;
        }

        // TODO: Once we move to `pushState()`, use the hash as
        // intended and use the `:target` selector so we don't
        // need to manually add/remove a class.

        this.$(".comment-list.top-level .linked").removeClass(linkedClass);
        $comment.addClass(linkedClass);

        var moveToComment = function () {
            $comment[0].scrollIntoView(true);
        };

        // Shift user to new link.
        moveToComment();

        // Also shift user when all images have downloaded.
        var $uncachedImgs = $(_.filter($(".content-container img"),
                function () {
                    return !this.complete;
                })),
            uncachedImgsLength = $uncachedImgs.length;

        $uncachedImgs.bind("load", function () {
            if (!--uncachedImgsLength) {
                moveToComment();
            }
        });

    },

    showNotFound: function () {
        this.notification.showMessage(AJS.I18n.getText("confluence.mobile.content.page.not.found"));
    },

    errorHandler: function (model, resp, options) {
        var that = this;

        if (resp.status == 404) {
            this.showNotFound();
            return;
        }

        options = _.extend(options, {
            message : AJS.I18n.getText("confluence.mobile.content.loading.error"),

            onRetry : function() {
                ConfluenceMobile.router.viewContent(that.contentId, that.commentId);
            },

            onCancel : function() {
                that.$el.show();
            }
        });

        this.notification.hide();
        ConfluenceMobile.genericAjaxErrorHandler(resp, model, options);
    },

    contentLink: function(event) {
        event.preventDefault();
        var a = $(event.target).closest("a");
        a[0].hash && ConfluenceMobile.Scrolling.resetByHash(a[0].hash);
        window.location = a.attr("href");
    }
});
