package com.atlassian.confluence.plugins.mentions;

import com.atlassian.confluence.content.render.xhtml.ConversionContext;
import com.atlassian.confluence.content.render.xhtml.DefaultConversionContext;
import com.atlassian.confluence.content.render.xhtml.Renderer;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.plugins.mentions.api.ConfluenceMentionEvent;
import com.atlassian.confluence.search.service.ContentTypeEnum;
import com.atlassian.confluence.setup.BootstrapManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.renderer.RenderContext;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.api.user.UserProfile;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.activation.MimetypesFileTypeMap;

import java.io.IOException;
import java.util.Set;

import static com.atlassian.confluence.core.BodyType.XHTML;

public class NotificationServiceImpl implements NotificationService {
    private static final MimetypesFileTypeMap fileTypeMap;
    private static final Logger log = LoggerFactory.getLogger(NotificationServiceImpl.class);

    static {
        fileTypeMap = new MimetypesFileTypeMap();
        try {
            fileTypeMap.addMimeTypes(IOUtils.toString(BootstrapManager.class.getResourceAsStream("/mime.types")));
        } catch (IOException exception) {
            throw new RuntimeException("Unable to load mime types", exception);
        }
    }

    private final EventPublisher eventPublisher;
    private final UserManager userManager;
    private final Renderer renderer;
    private final MentionsExcerptor mentionsExcerptor;

    public NotificationServiceImpl(final EventPublisher eventPublisher,
                                   final UserManager userManager,
                                   final Renderer renderer,
                                   final MentionsExcerptor mentionsExcerptor) {
        this.userManager = userManager;
        this.eventPublisher = eventPublisher;
        this.renderer = renderer;
        this.mentionsExcerptor = mentionsExcerptor;
    }

    @Override
    public void sendMentions(final Set<ConfluenceUser> mentionedUsers,
                             final ConfluenceUser author,
                             final ContentEntityObject content) {
        if (mentionedUsers.isEmpty()) {
            return;
        }

        final RenderContext renderContext = content.toPageContext();
        renderContext.setOutputType(RenderContext.EMAIL);

        final ConversionContext conversionContext = new DefaultConversionContext(renderContext);
        final ContentTypeEnum contentType = content.getTypeEnum();

        switch (contentType) {
            case COMMENT:
                for (final ConfluenceUser recipient : mentionedUsers) {
                    try (final AutoCloseable asUserCloseable = AuthenticatedUserThreadLocal.asUser(recipient)) {
                        final String commentMention = content.getBodyContent().getBodyType() == XHTML
                                ? renderer.render(content, conversionContext)
                                : null;
                        sendUserMention(recipient, author, content, commentMention);
                    } catch (Exception ex) {
                        log.error(
                                "Error sending comment mention. recipient: {}, author:{}, content id: {}",
                                recipient.getKey(),
                                author.getKey(),
                                content.getId(),
                                ex);
                    }
                }
                break;
            case PAGE:
            case BLOG:
            case CUSTOM:
                for (final ConfluenceUser recipient : mentionedUsers) {
                    final String excerpt = mentionsExcerptor.getExcerpt(content, recipient);
                    try (final AutoCloseable asUserCloseable = AuthenticatedUserThreadLocal.asUser(recipient)) {
                        final String mentionHtml = StringUtils.isNotBlank(excerpt)
                                ? renderer.render(excerpt, conversionContext)
                                : null;
                        sendUserMention(recipient, author, content, mentionHtml);
                    } catch (Exception ex) {
                        log.error(
                                "Error sending custom mention. recipient: {}, author:{}, content id: {}",
                                recipient.getKey(),
                                author.getKey(),
                                content.getId(),
                                ex);
                    }
                }
                break;
            default:
                break;
        }
    }

    private void sendUserMention(final ConfluenceUser recipient,
                                 final ConfluenceUser author,
                                 final ContentEntityObject content,
                                 final String mentionHtml) {
        final UserProfile recipientProfile = userManager.getUserProfile(recipient.getKey());
        final ConfluenceMentionEvent event = new ConfluenceMentionEvent(
                this,
                content,
                recipientProfile,
                author,
                mentionHtml
        );
        eventPublisher.publish(event);
    }
}
