package com.atlassian.confluence.plugins.mentions;

import com.atlassian.confluence.content.CustomContentEntityObject;
import com.atlassian.confluence.content.event.PluginContentCreatedEvent;
import com.atlassian.confluence.content.event.PluginContentUpdatedEvent;
import com.atlassian.confluence.core.BodyType;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.event.events.content.blogpost.BlogPostCreateEvent;
import com.atlassian.confluence.event.events.content.blogpost.BlogPostUpdateEvent;
import com.atlassian.confluence.event.events.content.comment.CommentCreateEvent;
import com.atlassian.confluence.event.events.content.comment.CommentUpdateEvent;
import com.atlassian.confluence.event.events.content.page.PageCreateEvent;
import com.atlassian.confluence.event.events.content.page.PageUpdateEvent;
import com.atlassian.confluence.pages.PageUpdateTrigger;
import com.atlassian.confluence.plugins.mentions.api.MentionFinder;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.security.access.ConfluenceAccessManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import java.util.Set;
import java.util.stream.Collectors;

import static com.atlassian.confluence.security.Permission.VIEW;

public class ConfluenceMentionsEventListener {

    private final EventPublisher eventPublisher;
    private final UserAccessor userAccessor;
    private final PermissionManager permissionManager;
    private final ConfluenceAccessManager accessManager;
    private final MentionFinder mentionFinder;
    private final NotificationService notificationService;

    public ConfluenceMentionsEventListener(final EventPublisher eventPublisher,
                                           final UserAccessor userAccessor,
                                           final PermissionManager permissionManager,
                                           final ConfluenceAccessManager accessManager,
                                           final MentionFinder mentionFinder,
                                           final NotificationService notificationService) {
        this.eventPublisher = eventPublisher;
        this.userAccessor = userAccessor;
        this.permissionManager = permissionManager;
        this.accessManager = accessManager;
        this.mentionFinder = mentionFinder;
        this.notificationService = notificationService;
    }

    @PostConstruct
    public void afterPropertiesSet() {
        eventPublisher.register(this);
    }

    @PreDestroy
    public void destroy() {
        eventPublisher.unregister(this);
    }

    @EventListener
    public void commentCreated(final CommentCreateEvent event) {
        sendNotificationsForNewContent(event.getComment());
    }

    @EventListener
    public void commentUpdated(final CommentUpdateEvent event) {
        sendNotificationsForUpdatedContent(event.getOriginalComment(), event.getComment());
    }

    @EventListener
    public void pageCreated(final PageCreateEvent event) {
        if (event.isSuppressNotifications() && event.getUpdateTrigger() == PageUpdateTrigger.LINK_REFACTORING) {
            return;
        }
        sendNotificationsForNewContent(event.getPage());
    }

    @EventListener
    public void pageUpdated(final PageUpdateEvent event) {
        //now we do intentionally send notifications to mentioned users on minor edits
        sendNotificationsForUpdatedContent(event.getOriginalPage(), event.getPage());
    }

    @EventListener
    public void blogPostCreated(final BlogPostCreateEvent event) {
        sendNotificationsForNewContent(event.getBlogPost());
    }

    @EventListener
    public void blogPostUpdated(final BlogPostUpdateEvent event) {
        //now we do intentionally send notifications to mentioned users on minor edits
        sendNotificationsForUpdatedContent(event.getOriginalBlogPost(), event.getBlogPost());
    }

    @EventListener
    public void pluginContentCreated(final PluginContentCreatedEvent event) {
        final CustomContentEntityObject content = event.getContent();
        if (content.getDefaultBodyType() == BodyType.XHTML) {
            sendNotificationsForNewContent(content);
        }
    }

    @EventListener
    public void pluginContentUpdated(final PluginContentUpdatedEvent event) {
        final ContentEntityObject content = event.getContent();
        if (content.getDefaultBodyType() == BodyType.XHTML) {
            sendNotificationsForUpdatedContent(
                    (ContentEntityObject) event.getOld(),
                    (ContentEntityObject) event.getNew()
            );
        }
    }

    private void sendNotificationsForNewContent(final ContentEntityObject content) {
        final Set<String> mentionedUserNames = mentionFinder.getMentionedUsernames(content.getBodyContent());
        sendContentNotifications(mentionedUserNames, content);
    }

    private void sendNotificationsForUpdatedContent(final ContentEntityObject originalEntity,
                                                    final ContentEntityObject newEntity) {
        /*
         * originalEntity might be null in case of X-updatedEvents
         * where content of the entity is not changed (and new version is not created).
         * Like page move when updating referral links.
         */
        if (originalEntity != null) {
            final Set<String> userNamesToNotify = mentionFinder.getNewMentionedUsernames(
                    originalEntity.getBodyContent(),
                    newEntity.getBodyContent()
            );
            sendContentNotifications(userNamesToNotify, newEntity);
        }
    }

    private Set<ConfluenceUser> getUsersWithViewPermission(final Set<String> userNames,
                                                           final ContentEntityObject content,
                                                           final ConfluenceUser author) {
        return userNames.stream()
                .map(userAccessor::getUserByName)
                .filter(user -> user != null
                        && (author == null || !author.equals(user))
                        && accessManager.getUserAccessStatus(user).hasLicensedAccess()
                        && permissionManager.hasPermissionNoExemptions(user, VIEW, content)
                )
                .collect(Collectors.toSet());
    }

    private void sendContentNotifications(final Set<String> mentionedUserNames,
                                          final ContentEntityObject content) {
        final ConfluenceUser contentAuthor = AuthenticatedUserThreadLocal.get();
        final Set<ConfluenceUser> usersWithPermission = getUsersWithViewPermission(mentionedUserNames, content, contentAuthor);
        notificationService.sendMentions(usersWithPermission, contentAuthor, content);
    }
}
