package com.atlassian.confluence.plugins.mentions.notifications;

import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.link.LinkType;
import com.atlassian.confluence.content.ui.ContentUiSupport;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.core.MaybeNot;
import com.atlassian.confluence.internal.ContentEntityManagerInternal;
import com.atlassian.confluence.notifications.Notification;
import com.atlassian.confluence.notifications.RenderContextProviderTemplate;
import com.atlassian.confluence.notifications.content.NotificationUserService;
import com.atlassian.confluence.pages.Comment;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.fugue.Either;
import com.atlassian.fugue.Maybe;
import com.atlassian.fugue.Option;
import com.atlassian.fugue.Pair;
import com.atlassian.plugin.notifications.api.medium.Message;
import com.atlassian.plugin.notifications.api.medium.NotificationAddress;
import com.atlassian.plugin.notifications.api.medium.ServerConfiguration;
import com.atlassian.plugin.notifications.api.medium.recipient.RoleRecipient;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.atlassian.sal.api.user.UserKey;

import java.util.HashMap;
import java.util.Map;

public class MentionRenderContextFactory extends RenderContextProviderTemplate<MentionContentPayload> {

    private final ContentEntityManagerInternal contentEntityManager;
    private final ContentUiSupport<ContentEntityObject> contentUiSupport;
    private final UserAccessor userAccessor;
    private final TransactionTemplate transactionTemplate;
    private final PermissionManager permissionManager;
    private final NotificationUserService notificationUserService;

    public MentionRenderContextFactory(final ContentEntityManagerInternal contentEntityManager,
                                       final ContentUiSupport<ContentEntityObject> contentUiSupport,
                                       final UserAccessor userAccessor,
                                       final TransactionTemplate transactionTemplate,
                                       final PermissionManager permissionManager,
                                       final NotificationUserService notificationUserService) {
        this.contentEntityManager = contentEntityManager;
        this.contentUiSupport = contentUiSupport;
        this.userAccessor = userAccessor;
        this.transactionTemplate = transactionTemplate;
        this.permissionManager = permissionManager;
        this.notificationUserService = notificationUserService;
    }

    @Override
    protected Maybe<Map<String, Object>> checkedCreate(final Notification<MentionContentPayload> mentionContentPayloadNotification,
                                                       final ServerConfiguration configuration,
                                                       final Maybe<Either<NotificationAddress, RoleRecipient>> roleRecipient) {
        if (roleRecipient.isEmpty() || roleRecipient.get().isLeft()) {
            return MaybeNot.becauseOf(
                    "This factory exposes content, thus recipient has to be provided in order to perform a VIEW permission check.");
        }

        final RoleRecipient recipient = roleRecipient.get().right().get();
        final ConfluenceUser recipientUser = userAccessor.getUserByKey(recipient.getUserKey());

        final HashMap<String, Object> renderContext = new HashMap<>();
        final MentionContentPayload payload = mentionContentPayloadNotification.getPayload();

        if (payload.getMentionHtml().isDefined()) {
            renderContext.put("contentHtml", payload.getMentionHtml().get());
        }
        final Option<Pair<Content, String>> maybeContentForId = getContentDetailsForId(payload.getContentId(), recipient.getUserKey());

        if (maybeContentForId.isEmpty()) {
            return MaybeNot.becauseOf("Unable to load content with id " + payload.getContentId() + " may not permission"
                    + "to view the content");
        }

        final Pair<Content, String> contentForId = maybeContentForId.get();
        final Content content = contentForId.left();
        final String contentType = contentForId.right();

        renderContext.put("content", content);
        renderContext.put("contentType", contentType);
        if (Comment.CONTENT_TYPE.equals(contentType)) {
            renderContext.put(Message.MESSAGE_ID, String.valueOf(((Content)content.getContainer()).getId().asLong()));
        } else {
            renderContext.put(Message.MESSAGE_ID, String.valueOf(content.getId().asLong()));
        }

        renderContext.put("modifier", notificationUserService.findUserForKey(recipientUser, payload.getAuthorUserKey()));

        return Option.some(renderContext);
    }

    /**
     * Construct a content object from an arbitrary id
     * TODO: Replace with a content service call when the Content API supports pluggable content types
     *
     * @return A content object with the id, title and web ui populated
     * @deprecated since 1.9.4
     */
    @Deprecated
    private Option<Pair<Content, String>> getContentDetailsForId(final long id, final UserKey recipient) {
        return transactionTemplate.execute(() -> {
            final ContentEntityObject content = contentEntityManager.getById(id);

            final ConfluenceUser user = userAccessor.getUserByKey(recipient);

            if (permissionManager.hasPermissionNoExemptions(user, Permission.VIEW, content)) {
                //ContentId does not store the content type or use it
                //  so we are safe enough using page as the content type atm
                //  This should be fixed once the content API fixes this
                //TODO: CONFDEV-25152
                final ContentId contentId = ContentId.of(ContentType.PAGE, id);
                return Option.some(Pair.pair(Content.builder()
                        .id(contentId)
                        .title(content.getDisplayTitle())
                        .addLink(LinkType.WEB_UI, content.getUrlPath())
                        .build(), contentUiSupport.getContentTypeI18NKey(content)));
            } else {
                return Option.none();
            }
        });
    }
}
