define('confluence/cjc/create-issue-form/create-issue-form-view-abstract', [
    'ajs',
    'jquery',
    'backbone',
    'confluence/cjc/issue/model',
    'confluence/cjc/issue/collection',
    'confluence/cjc/issue-field/model',
    'confluence/cjc/create-issue-form-submit-validation'
],
function(
    AJS,
    $,
    Backbone,
    IssueModel,
    IssueCollection,
    IssueFieldModel,
    CreateIssueValidation
) {
    'use strict';

    /**
     * It is a parent view for creating single and multiple issues views.
     */
    var CreateIssueFormView = Backbone.View.extend({

        initialize: function(options) {
            var self = this;

            this.formObjectView = options.formObjectView;

            this.mappingFields = this.formObjectView.mappingFieldCollection;
            this.selectionObject = this.formObjectView.selectionObject;

            this.createIssueFormValidation = new CreateIssueValidation({
                $form: this.$el
            });

            this.listenTo(this.formObjectView, 'rendered', function() {
                this.render();
            }, this);

            // update UI basing on changing value of `status` property of a field
            this.listenTo(this.mappingFields, 'change:status', function(fieldModel, changedValue) {
                var fieldName = fieldModel.get('name');
                var isHidden = changedValue === window.Confluence.CreateJiraContent.JiraIssue.FieldStatus.HIDDEN;
                self.getWrapperFieldElement(fieldName).toggleClass('hidden', isHidden);
            }, this);

            // update UI basing on changing value of `required` property of a field
            this.listenTo(this.mappingFields, 'change:required', function(fieldModel, isRequired) {
                var fieldName = fieldModel.get('name');
                var $field = self.getWrapperFieldElement(fieldName);

                if (isRequired) {
                    if ($field.find('select').val() === -1) {
                        $field.find('select')
                                    .val($field.find("option:not('.hidden'):first").val())
                                    .change();
                    }
                }

                $field.find("option[value='-1'").addClass('hidden', !isRequired);
                $field.find('label > span').toggleClass('hidden', !isRequired);
            }, this);
        },

        render: function() {
            var self = this;

            this.listenTo(this.formObjectView, 'confluence-jira-content.form-updated-fieldmeta', function(metaFields) {
                if (metaFields) {
                    self._handleRequiredField(metaFields);
                }
            });
        },

        /**
         * Get issues collection
         * @returns {IssueCollection}
         */
        getIssueCollection: function() {
            throw new Error('CreateIssueForm.getIssueCollection is not implemented. It should be implemented in child view class.');
        },

        /**
         * Collect custom fields which are renders by Jira Integration plugin
         * @param {IssueModel} issueModel
         */
        extractExtraFields: function(issueModel) {
            this.$('#jira-required-fields-panel')
                .find('.jira-field').each(function(i, el) {
                    var $this = $(el);

                    var fieldName = $this.attr('data-jira-type');
                    var field = this.getFieldObjectFromDOM(fieldName, $this);
                    issueModel.get('fields').add(field);

                }.bind(this));
        },

        /**
         * Try to get input value by users.
         *
         * There are 2 parts/ares which are rendering in the dialog:
         * - Render by our plugin, Jira-Create-Content
         *      + We supports 2 fields, ex: so far, they are Summary and Description fields.
         * - Render by JIP plugin,
         *      + Remaining fields, except Summary and Description fields.
         * Please take a look at `.../issue-field/init.js` to see all supported fields.
         *
         * @param {string} fieldName - for custom fields, the `fieldName` can be like that: `com.atlassian.jira.plugin.system.customfieldtypes:textfield`.
         * @param {jQuery object} $elFieldGroup - a wrapper DOM of the input
         * @returns {*}
         * @private
         */
        getFieldObjectFromDOM: function(fieldName, $elFieldGroup) {
            var val = $.trim($elFieldGroup.find('input[name], select[name], textarea[name]').val());
            var isRequired = $elFieldGroup.find('.icon-required').is(':visible');
            var fieldModel = this.mappingFields.getIssueFieldByName(fieldName);

            if (fieldModel) {
                // in supported field
                fieldModel = fieldModel.clone();
                fieldModel.set({
                    content: val,
                    required: isRequired
                });
            } else {
                // not in supported field
                fieldModel = new IssueFieldModel({
                    name: fieldName,
                    required: isRequired,
                    content: val,
                    label: $elFieldGroup.find('label').text(),
                    isSupport: false
                });
            }

            fieldModel.set('index', -1);
            return fieldModel;
        },

        /**
         * Toggle showing and hiding of switching single and multple creating issues link.
         * @param {boolean} isGoToSingle - going to creating single issue view.
         */
        toggleLinkSwitchSingleMultipleForm: function(isGoToSingle) {
            var $btnSwitching = this.$('.js-text-suggestion');

            var text = isGoToSingle
                    ? AJS.I18n.getText('createjiracontent.dialog.form.create.single.issue')
                    : AJS.I18n.getText('createjiracontent.dialog.form.create.multiple.issues');

            $btnSwitching
                    .toggleClass('js-go-to-create-multiple-issues', !isGoToSingle)
                    .toggleClass('js-go-to-create-single-issue', isGoToSingle)
                    .text(text)
                    .removeClass('hidden');
        },

        renderErrorMessages: function(errors) {
            if (errors.length > 0 && errors[0].type) {
                this.formObjectView.errorMessageView.removeMessage(errors[0].type);
                this.formObjectView.errorMessageView.addMessage(errors);
            }
        },

        _handleRequiredField: function(metaFields) {

            // handle special case for description, because description do not exist in `requiredFields` (for JIP integration)
            var isRequired = metaFields.descriptionFieldStatus === Confluence.CreateJiraContent.JiraIssue.FieldStatus.REQUIRED;
            this.mappingFields.setFieldRequired('description', isRequired);

            // `status` property is to determine hidden or not.
            var descMappingField = this.mappingFields.getIssueFieldByName('description');
            descMappingField.set('status', metaFields.descriptionFieldStatus);

            var fields = this.mappingFields.filter(function(issueFieldModel) {
                var omitList = ['summary', 'description'];
                if (omitList.indexOf(issueFieldModel.get('name')) >= 0) {
                    return false;
                }
                return true;
            });

            _.each(fields, function(issueFieldModel) {
                var isRequired = metaFields.requiredFields[issueFieldModel.get('name')] !== null;
                issueFieldModel.set('required', isRequired);
            });

            this.formObjectView.refreshDialog();
        },

        /**
         * Get DOM wrapper of a field by name
         * @param fieldName - field name
         * @returns {jQuery object}
         */
        getWrapperFieldElement: function(fieldName) {
            var $field;

            if (this.formObjectView.isInTextMode()) {
                // in creating single issue view
                $field = this.$(AJS.format('#issue-{0}', fieldName));
            } else {
                // in creating multiple issues view
                $field = this.$(AJS.format('#issue-{0}-index', fieldName));
            }

            return $field.parent();
        },

        submit: function() {
            this.createIssueFormValidation.startValidation(
                this.submitFormWhenValid.bind(this),
                this.submitFormWhenInvalid.bind(this)
            );
        },

        submitFormWhenValid: function() {
            throw new Error('submitWhenValid: not yet implemented');
        },

        submitFormWhenInvalid: function(errors) {
            throw new Error('submitWhenValid: not yet implemented');
        }
    });

    return CreateIssueFormView;
});
