/**
 * Autocomplete appears when you press a trigger characters (//) in the editor.
 */
(function($, tinymce) {
    "use strict";

    /*
     * Some constant strings
     * */
    var MCE_PLUGIN_KEY = 'dateautocomplete',
        MCE_PLUGIN_NAME = 'tinymce.plugins.DateAutocomplete',
        COMMAND_NAME = "mceConfInsertDateAutocomplete",
        MENU_COMMAND_NAME = "confMenuInsertDate",
        KEY_TRIGGER = "//";

    // keeping it camelCase for consistency within confluence files for easier search and replace
    var synchronyId = 'confluence.date-autocomplete.plugin';

    /*
     * Some global varibales
     * */
    var InlineTasksUtil = Confluence.InlineTasks.Util,
        NODE_TYPE = InlineTasksUtil.NODE_TYPE,
        KEY = InlineTasksUtil.KEY,
        DateUtil = Confluence.InlineTasks.DateUtil,
        editor = null, //just want to cache editor and initialize when init plugin
        dateAutoComplete = null, // is instance of AutoComplete.Manager
        isMoveRight = false,
        TreeWalker = tinymce.dom.TreeWalker;

    var Util = {
        /**
         * Init Date AutoComplete TinyMce plugin
         * 1. Add some settings in Confluence.Editor.Autocompleter.Settings
         * 2. Initialize Confluence.Editor.AutoComplete.Manager
         */
        initDateAutoCompleteObject: function(ed) {
            // Confluence 4.1.x uses tinymce.confluence, 4.2 onwards use Confluence.Editor
            if (!Confluence.Editor.Autocompleter) {
                Confluence.Editor.Autocompleter = tinymce.confluence.Autocompleter;
            }

            // Date Autocomplete settings.
            Confluence.Editor.Autocompleter.Settings[KEY_TRIGGER] = {

                ch : KEY_TRIGGER,
                endChars : [],

                update : function(autoCompleteControl, data) {
                    dateAutoComplete.picker.die();

                    // if user hasn't modify the placeholder, insert the current date
                    var date = (data == DateUtil.PATTERN_INSERTING.toLowerCase())
                            ? moment() : DateUtil.guessPartialDate(data, DateUtil.PATTERN_INSERTING);
                    date && DateUtil.insertDateIntoCurrentCursor(date, "<time>", "", DateUtil.PATTERN_LOZENGE, "&nbsp;");

                    if (dateAutoComplete.dateInserted()) {
                        AJS.trigger('analyticsEvent', {name: 'confluence-spaces.tasks.daterecognition.used'});
                    }

                    //CONFDEV-2263: must kill dateAutoComplete if not, some key handles of dateAutoComplete is still remaining
                    dateAutoComplete.reset();

                    ed.undoManager.add();

                    AJS.trigger('synchrony.start', { id: synchronyId });
                }

            };

            var settingsOfAutoCompleteManager = {
                leadingChar: KEY_TRIGGER, //leadingChar must the same as key trigger

                isDataValid: function(data) {
                    return !dateAutoComplete.dirty() || DateUtil.guessPartialDate(data, DateUtil.PATTERN_INSERTING) != null;
                },

                onBeforeDie: function() {
                    dateAutoComplete.picker && dateAutoComplete.picker.die();

                    //after inserting date lozenge, user press Escape or click outside after that to cancel
                    //we will delete current date text was inserted before
                    if (dateAutoComplete.control) {
                        var $container = $(dateAutoComplete.control.getContainer());
                        if (!dateAutoComplete.dirty()) {
                            $container.find("#autocomplete-search-text").remove();
                        }
                    }
                },

                onAfterStart: function(option){
                    if (option && option.date) {
                        Util.fillPlaceholderDateAutoComplete(ed, option.date, true);
                        dateAutoComplete.dirty(true);
                        dateAutoComplete.dateInserted(true);
                    } else if (option && !option.isTriggerFromOrphan) {
                        //trigger start autocomplete from fresh
                        Util.fillPlaceholderDateAutoComplete(ed, DateUtil.PATTERN_INSERTING.toLowerCase(), false);
                        dateAutoComplete.dateInserted(false);
                    }

                    var dateStr = dateAutoComplete.control.text();

                    var objStartDate = DateUtil.guessPartialDate(dateStr, DateUtil.PATTERN_INSERTING) || moment();
                    Util.bindDatePickerToAutoComplete(objStartDate);
                },

                onScroll: function(){
                    dateAutoComplete.picker && dateAutoComplete.picker.placeDatePicker();
                }
            };

            dateAutoComplete = new Confluence.Editor.AutoComplete.Manager(settingsOfAutoCompleteManager);
        },

        /**
         * Active Date Autocomplete
         * @param insertedDate date object is inserted when trigger Date Autocomplete, if insertedDate is not passed
         *        current date is inserted by default.
         * @returns {boolean}
         */
        activateDateAutocomplete: function(ui, val, options) {
            var allOptions = Object.assign({}, options, val); // We only pass a single object to AutocompleteManager.start
            if (!allOptions.hasOwnProperty('insertedFromMenu')) {
                allOptions['insertedFromMenu'] = false;
            }
            //do not allow insert more than one date complete date at a same time
            if (!dateAutoComplete.isAlive()) {
                var isCreated = dateAutoComplete.start(allOptions);
                if (!isCreated && !allOptions['insertedFromMenu']) {
                    //re-insert back '//' to current position
                    editor.execCommand("mceInsertContent", false, KEY_TRIGGER, {skip_undo: true});
                    return false;
                }
            }
        },

        /**
         * Bind date picker to current span#autocomplete
         * @param {String} objStartDate a date object to set a start date for date picker
         */
        bindDatePickerToAutoComplete: function(objStartDate){
            var $spanAutoComplete = dateAutoComplete.control && dateAutoComplete.control.getContainer();

            if(!$spanAutoComplete.length) {
                return;
            }

            $spanAutoComplete.addClass("date-autocomplete");
            $spanAutoComplete.find("#autocomplete-search-text span").addClass("inserting");

            var onSelect = function(date) {
                    dateAutoComplete.control.update(date.format(DateUtil.PATTERN_INSERTING));
                },
                isSetDueDate = Util.isInsideTaskAndFirstTimeNode($spanAutoComplete);

            dateAutoComplete.picker = DateUtil.datepicker.create({
                $attachTo: $spanAutoComplete,
                $positionTo: $spanAutoComplete,
                startDate: objStartDate,
                onSelect: onSelect,
                isSetDueDate: isSetDueDate
            });
        },

        /**
         * Write the placeholder and select (highlight) it, so that users can start typing to replace
         * @param ed
         * @param text
         */
        fillPlaceholderDateAutoComplete: function(ed, text, collapse) {
            var inserting = dateAutoComplete.control.getContainer().find("#autocomplete-search-text span");
            inserting.text(AJS.Rte.HIDDEN_CHAR + text);

            var r = ed.dom.createRng();
            r.setStart(inserting[0].firstChild, 1);
            r.setEnd(inserting[0].firstChild, text.length + 1);
            collapse && r.collapse(false);
            ed.selection.setRng(r);
        },

        autocompleteDoubleSlash: function(e, data) {
            var handlerManager = data.handlerManager,
                ed = data.ed,
                createHandler = data.createHandler;

            var handlerDoubleSlash = createHandler(
                /(?:\s|\xA0|^|\ufeff|\u200b)(\/)$/,
                function() {
                    ed.execCommand(COMMAND_NAME, false, {}, {skip_undo: true});
                },
                true, //swallow trigger char
                "#autocomplete" //do not trigger inside #autocomplete
            );

            handlerManager.registerHandler("/".charCodeAt(0), handlerDoubleSlash);
        },

        /**
         * Register the command to active date autocomplete
         * when user type double slashes "//" in editor
         * @param ed
         */
        registerTriggerDoubleSlash: function(ed) {
            // Register the command so that it can be invoked by using tinymce.activeEditor.execCommand(COMMAND_NAME);
            ed.addCommand(COMMAND_NAME, Util.activateDateAutocomplete);

            ed.addCommand(MENU_COMMAND_NAME, function () {
                AJS.trigger('analytics', {name: 'editor.insert-data-from-menu'});
                editor.execCommand('mceConfInsertDateAutocomplete', false, {insertedFromMenu: true}, {skip_undo: true});
            }.bind(this));

            //register handler for trigger '//'
            AJS.bind("confluence.editor.registerHandlers", this.autocompleteDoubleSlash);
        },

        /**
         * Check a node is time node or not
         * @param node
         * @returns {boolean} true if the node is time node, or not return false
         */
        isTimeNode: function(node) {
            return node &&
                        node.nodeType === NODE_TYPE.ELEMENT_NODE &&
                        node.tagName.toLowerCase() === "time";
        },

        /**
         * At the cursor, try to get time node
         * @param ed tinymce editor object
         * @returns DOM node if find time not, or not, return null
         */
        getTimeNodeAtCursor: function(ed) {
            if (!ed) {
                ed = editor;
            }
            var nodeTime = ed.selection.getNode();
            if  (Util.isTimeNode(nodeTime)) {
                return nodeTime;
            }

            nodeTime = ed.selection.getStart();
            if  (Util.isTimeNode(nodeTime)) {
                return nodeTime;
            }

            nodeTime = ed.selection.getEnd();
            if  (Util.isTimeNode(nodeTime)) {
                return nodeTime;
            }
        },

        makeTimeNodesUnEditable: function($times) {
            $times.each(function() {
                var $this = $(this);

                //remove empty time element when pasting
                if ($.trim($this.text()) === "") {
                    $this.remove();
                }

                // in Chrome, when Copy & Paste quickly and many times or in some edge cases,
                // pasted content has time nodes which have <br class="Apple-interchange-newline"> inside
                tinymce.isWebKit && $this.remove('br.Apple-interchange-newline');

                InlineTasksUtil.addContentEditableIfApplicable(this);
            });
        },

        /**
         * CONFDEV-22419: <time> node wrapping around with HIDDEN_CHAR makes UP/DOWN keys works correctly
         * Always make sure there are text nodes around time node
         */
        wrapTimeNodesWithHiddenChar: function($times) {
            $times.each(function() {
                var $node = $(this);

                if (!this.previousSibling ||
                    this.previousSibling.nodeType !== NODE_TYPE.TEXT_NODE ||
                    this.previousSibling.nodeValue === "") {
                    $node.before(AJS.Rte.HIDDEN_CHAR);
                }

                if (!this.nextSibling ||
                    this.nextSibling.nodeType !== NODE_TYPE.TEXT_NODE ||
                    this.previousSibling.nodeValue === "") {
                    $node.after(AJS.Rte.HIDDEN_CHAR);
                }
            });
        },

        /**
         * Put cursor after or before a node
         * @param ed editor object
         * @param node
         * @param isAfter true is after, opposite is false
         */
        putCursorAtEdge: function(ed, node, isAfter) {
            isAfter = !!isAfter;

            if (isAfter) {
                if (!node.nextSibling || node.nextSibling.nodeType !== NODE_TYPE.TEXT_NODE) {
                    $(node).after(ed.dom.doc.createTextNode(""));
                }
            }
            else {
                if (!node.previousSibling || node.previousSibling.nodeType !==  NODE_TYPE.TEXT_NODE) {
                    $(node).before(ed.dom.doc.createTextNode(""));
                }
            }

            var r = ed.selection.dom.createRng();
            r.selectNode(isAfter ? node.nextSibling : node.previousSibling);
            r.collapse(isAfter);
            ed.selection.setRng(r);
        },

        findFirstTimeNodeInClosestBlock: function(ed, n, isForward) {
            var walker = new TreeWalker(n, ed.getBody());
            var orgNode = n, currentBlock = 0;
            do {
                //only look for element in the next block
                if (currentBlock == 1 && Util.isTimeNode(n))
                    return n;

                //count block elements
                //every time we see a block element with sibling, that means we have a new hard line
                var lastSeenSibling = isForward ? n.previousSibling : n.nextSibling;
                if (n !== orgNode && lastSeenSibling && (ed.dom.isBlock(n) || ed.dom.isBlock(lastSeenSibling)))
                    currentBlock++;
            } while ((n = isForward ? walker.next() : walker.prev())
                && currentBlock < 2);
        },

        findFirstTimeNodeNearby: function(ed, n, isForward) {
            var orgNode = n;
            var walker = new TreeWalker(n, ed.getBody());
            do {
                if (Util.isTimeNode(n)) {
                    return n;
                }
                // if encounter non-empty text or block element, don't need to search further
                if (ed.dom.isBlock(n) ||
                   (n !== orgNode && n.nodeType === NODE_TYPE.TEXT_NODE && n.nodeValue !== "")) {
                    return;
                }
            } while (n = isForward ? walker.next() : walker.prev());
        },

        /**
         * Convert <time> node whose datetime attribute is invalid to plain text
         * Accepted datetime attribute is always YYYY-MM-DD
         * @param $times
         */
        convertInvalidTimeNodeToPlainText: function($times){
            $times.each(function() {
                var $this = $(this);

                var strDate = $this.attr("datetime");
                var objDate = DateUtil.parse(strDate);
                if (!objDate) {
                    $this.before($this.text());
                    $this.remove();
                }
            });
        },

        /**
         * Check whether el is inside a task list and a the first one of the task list
         * @param el is able to be time element or span.date-autocomplete
         * @returns {boolean}
         */
        isInsideTaskAndFirstTimeNode: function(el){
            var $el = $(el),
                $li = $el.closest("ul.inline-task-list > li[data-inline-task-id]");

            //check inside task list first
            if(!$li.length) {
                return false;
            }

            return $li.find("time, span.date-autocomplete")[0] === $el[0];
        }
    };

    var HandleEvent = {
        /**
         * In FireFox do now allow to delete an element which has 'contenteditable' attribute
         * So we need to use script to delete
         * @param e
         */
        handleRemoveDateLozenge: function(e) {
            var keyCode = e.keyCode;


            if (keyCode !== KEY.BACKSPACE &&
                keyCode !== KEY.DELETE) {
                return;
            }

            var isForward = (keyCode === KEY.DELETE),
                range = editor.selection.getRng(true),
                startNode = range.startContainer,
                node = startNode;

            if (startNode.nodeType == NODE_TYPE.TEXT_NODE &&
                (isForward ? range.startOffset != startNode.nodeValue.length : range.startOffset != 0))
                    return;

            if (startNode.nodeType === NODE_TYPE.ELEMENT_NODE) {
                if (isForward)
                    node = (startNode.childNodes.length === range.startOffset) ? startNode : startNode.childNodes.item(range.startOffset);
                else
                    node = (range.startOffset === 0) ? startNode : startNode.childNodes.item(range.startOffset - 1);
            }

            var timeNode = Util.findFirstTimeNodeNearby(editor, node, isForward);
            timeNode && editor.dom.remove(timeNode);
        },

        handleClickingDateLozenge: function(e) {
            var $target = $(e.target);
            if (! $target.is("time")) {
                return;
            }

            AJS.trigger('synchrony.stop', { id: synchronyId });

            // Fire an analytics event when the date lozenge is clicked in the editor.
            var context = ($target.closest("li[data-inline-task-id]").length)  ? 'task' : 'page';
            var data = {mode: 'editor', context: context};
            AJS.trigger('analyticsEvent', {name: 'confluence-spaces.date.clicked', data : data});

            var $iframe = $target.closest("body");
            var closeControl = function() {
                AJS.Rte.unbindScroll("date-lozenge-date-picker");
                picker.die();
                AJS.trigger('synchrony.start', { id: synchronyId });
            };

            $iframe.one("keydown click", closeControl);

            var onSelect = function(date) {
                var datetimeFormatted = date.format(DateUtil.PATTERN_INSIDE_TIME_ELE);
                var lozengeFormatted = date.format(DateUtil.PATTERN_LOZENGE);

                // If the date has changed, fire an analytics event.
                if (datetimeFormatted !== $target.attr("datetime")) {
                    var data = {context: context};
                    AJS.trigger('analyticsEvent', {name: 'confluence-spaces.date.changed', data : data});
                }

                $target.attr("datetime", datetimeFormatted);
                $target.text(lozengeFormatted);
                $iframe.unbind("keydown click", closeControl);

                AJS.trigger('synchrony.start', { id: synchronyId });
            };

            var startDate = DateUtil.parse($target.attr("datetime")),
                isSetDueDate = Util.isInsideTaskAndFirstTimeNode($target),
                picker = DateUtil.datepicker.create({
                    $attachTo: $target,
                    $positionTo: $target,
                    startDate: startDate,
                    onSelect: onSelect,
                    isSetDueDate: isSetDueDate
                });

            var onScroll = _.bind(_.throttle(function(){
                if (picker && AJS.Rte.isAnyPartShown($target)) {
                    picker.placeDatePicker();
                }
                else {
                    closeControl();
                }
            }, 40), this);

             //CONFDEV-22643 - reposition of date-picker when scrolling
            AJS.Rte.bindScroll("date-lozenge-date-picker", onScroll);
        },

        handleTypingDate: function(e) {
            var keyCode = e.keyCode;

            //do not need to check valid date in lozenge date for navigation keys
            if (keyCode === KEY.UP ||
                keyCode === KEY.DOWN ||
                keyCode === KEY.LEFT ||
                keyCode === KEY.RIGHT ||
                keyCode === KEY.HOME ||
                keyCode === KEY.END ||
                keyCode === KEY.PAGEDOWN ||
                keyCode === KEY.PAGEUP ||
                e.metaKey || e.ctrlKey || e.altKey) {
                return;
            }

            if (dateAutoComplete.control && dateAutoComplete.picker) {
                var dateStr = dateAutoComplete.control.text();
                dateAutoComplete.picker.setDate(dateStr);
            }
        },

        /**
         * When content of editor is loaded, make sure all <time> elements have some necessary attributes
         * */
        handleOnLoadContent: function() {
            var $doc = $(editor.dom.doc),
                $times = $doc.find("time");

            Util.makeTimeNodesUnEditable($times);
            Util.wrapTimeNodesWithHiddenChar($times);
        },

        handlePostPaste: function(e, pl, o) {
            Util.convertInvalidTimeNodeToPlainText($(o.node).find("time"));
        },

        handlePostPasteContent: function(e, pl, o) {
            var $times = $(o.node).find("time");
            Util.makeTimeNodesUnEditable($times);
        },

        handlePrePasteContent: function(e, pl, o) {
            if (Util.getTimeNodeAtCursor(editor)) {
                o.content = "";
                return false;
            }
        },

        /**
         * CONFDEV-22419
         * In order to navigation via UP/DOWN keys works naturally,
         * the function makes sure <li> or <p> should not have only time node.
         * <li> or <p> should have text node around time node.
         *
         * @param e event object of key event
         */
        fixWhenUpDownNavOnList: function(e) {
            var key = e.keyCode;

            if (key !== KEY.UP &&
                key !== KEY.DOWN) {
                return;
            }

            var range = editor.selection.getRng(true),
                startNode = range.startContainer,
                node = startNode,
                isForward = key === KEY.DOWN;

            if (startNode.nodeType === NODE_TYPE.ELEMENT_NODE){
                if (isForward)
                    node = (startNode.childNodes.length === range.startOffset) ? startNode : startNode.childNodes.item(range.startOffset);
                else
                    node = (range.startOffset === 0) ? startNode : startNode.childNodes.item(range.startOffset - 1);
            }

            var timeNode = Util.findFirstTimeNodeInClosestBlock(editor, node, isForward);
            if (timeNode) {
                Util.wrapTimeNodesWithHiddenChar($(timeNode));
            }
        },

        /**
         * Prevent cursor going inside date lozenge
         *
         * @param o current changed node in onNodeChanged event
         */
        preventCursorInsideDateLozenge: function(o) {
            var el = (o.element.nodeType === NODE_TYPE.TEXT_NODE) ? o.element.parentNode : o.element;

            if (Util.isTimeNode(el)) {
                //jump cursor after time node
                if (isMoveRight || editor.selection.getRng().startOffset > 1) {
                    Util.putCursorAtEdge(editor, el, true);
                } else { //jump cursor before time node
                    Util.putCursorAtEdge(editor, el, false);
                }
            }
        },

        //CONFDEV-22605 - Handle undo/redo correctly - reattach autocomplete, if needed
        handleUndoRedo: function(){
            dateAutoComplete && dateAutoComplete.reattach();
        }
    };

    /**
    * Some necessary options to create Tinymce DateAutoComplete plugin
    * */
    var tinymcePluginCreatingOption = {
        init : function(ed) {
            Util.initDateAutoCompleteObject(ed);
            Util.registerTriggerDoubleSlash(ed);

            //cache editor to re-use somewhere
            editor = ed;

            // destroy Date AutoComplete when a context menu is shown
            ed.on('contextmenu', function() {
                dateAutoComplete.reset();
            });

            //handle when user press edit buttom from viewmode
            ed.on('SetContent', HandleEvent.handleOnLoadContent);

            ed.on('click', HandleEvent.handleClickingDateLozenge);
            ed.on('keyup', HandleEvent.handleTypingDate);

            if (tinymce.isWebKit) {
                ed.on('keydown', HandleEvent.fixWhenUpDownNavOnList);
                ed.on('init', function() {
                    //in Webkit, when Copy & Paste, a new time node do not have contenteditable=false
                    // so we should add it manually
                    $(document).bind('postPaste', HandleEvent.handlePostPasteContent);
                });
                ed.on('remove', function() {
                    $(document).unbind('postPaste', HandleEvent.handlePostPasteContent);
                });
            }

            ed.on('keyup', function(e) {
                isMoveRight = (e.keyCode === KEY.RIGHT);

                // Date autocomplete without the // prelude:
                // Check for dates so we can pop the date picker if a date-like string is input.
                var range = ed.selection.getRng(true);
                var ancestor = range.commonAncestorContainer;
                var textContent = !!ancestor && ancestor.data;

                if (!!textContent && !AJS.$(ancestor).closest("pre,.text-placeholder").length) {
                    // Grab the last date-length string.
                    var text = String.prototype.slice.apply(textContent, [-11]);

                    // Check if the user has typed a digit. We do not want this to trigger
                    // when navigation keys etc are pressed.
                    if ((e.keyCode >= 48 && e.keyCode <= 57) || (e.keyCode >= 96 && e.keyCode <= 105)) {

                        var dateRegex = /(^| )\d{2}[\/\-]\d{2}[\/\-]\d{4}$/;
                        var date = String.prototype.slice.apply(textContent, [-10]);

                        if (dateRegex.test(textContent) && DateUtil.parse(date, [DateUtil.PATTERN_INSERTING, DateUtil.PATTERN_INSERTING_ALTERNATE])) {
                            var node = range.commonAncestorContainer;
                            // Select the date string so that it is replaced by autoformat.
                            range.setStart(node, range.endOffset - date.length);

                            ed.undoManager.add();
                            ed.undoManager.beforeChange();
                            ed.selection.setRng(range);
                            ed.execCommand("mceConfInsertDateAutocomplete", false, {date: date}, {skip_undo: true});

                            AJS.trigger('analyticsEvent', {name: 'confluence-spaces.tasks.daterecognition.triggered'});
                        }
                    }
                }

            });

            if (tinymce.isGecko) {
                ed.on('keydown', HandleEvent.handleRemoveDateLozenge);
                ed.on('init', function() {
                    //in Firefox, user can use some tricks to put cursor inside <time> node
                    //so we also need to prevent paste content when cursor inside <time> node
                    $(document).bind('prePaste', HandleEvent.handlePrePasteContent);
                });
                //CONFDEV-30102: unbind event to make sure handler will not run when s.o does not include this plugin into RTE
                ed.on('remove', function() {
                    $(document).unbind('prePaste', HandleEvent.handlePrePasteContent);
                });
            }

            if (tinymce.isWebKit || tinymce.isGecko) {
                ed.on('NodeChange', HandleEvent.preventCursorInsideDateLozenge);
            }

            //CONFDEV-22605 - Handle undo/redo correctly - reattach autocomplete, if needed
            ed.on('Undo', HandleEvent.handleUndoRedo);
            ed.on('Redo', HandleEvent.handleUndoRedo);

            //CONFDEV-23988
            ed.on('init', function() {
                $(document).bind('postPaste', HandleEvent.handlePostPaste);
            });
            //CONFDEV-30102: clean all event handler
            ed.on('remove', function() {
                $(document).unbind('postPaste', HandleEvent.handlePostPaste);

                //un-register handler for trigger '//'
                AJS.unbind("confluence.editor.registerHandlers", Util.autocompleteDoubleSlash);
            });

            /**
             * CONFDEV-47416: remove contenteditable from synchrony whitelist
             * means contenteditable is not synchronized across users
             * => listen to remote change to update contenteditable
             */
            AJS.bind("editor.remote.change", function () {
                Util.makeTimeNodesUnEditable($(ed.dom.doc).find("time"));
            });
        },

        getInfo : function() {
            return {
                longname : 'Insert Date Autocomplete',
                author : 'Atlassian',
                authorurl : 'http://www.atlassian.com',
                version : tinymce.majorVersion + "." + tinymce.minorVersion
            };
        }
    };

    /**
    * This is entrance point of Tinymce Date AutoComplete plugin
    * */
    function initTinyMcePlugin() {
        tinymce.create(MCE_PLUGIN_NAME, tinymcePluginCreatingOption);

        // Register plugin
        tinymce.PluginManager.add(MCE_PLUGIN_KEY, tinymce.plugins.DateAutocomplete);

        AJS.Rte.BootstrapManager.addTinyMcePluginInit(function(settings) {
            settings.plugins += "," + MCE_PLUGIN_KEY;
        });
    }

    initTinyMcePlugin();

})(AJS.$, tinymce);
