/**
 * Collect all static functions in one place.
 * Because Date library can be changed in future, so we want all code to depend on Confluence.InlineTasks.DateUtil
 * Date Library is used: http://momentjs.com/
 */
define('confluence-inline-tasks/dateutil', [
    'jquery',
    'ajs',
    'moment',
    'confluence/position',
    'confluence-inline-tasks/util',
    'aui/inline-dialog2',
    'confluence-inline-tasks/date-picker1'
], function (
    $,
    AJS,
    moment,
    Position,
    InlineTasksUtil
) {
    'use strict';

    var DateUtil = {
        /*
         * Some constant date patterns used in this plugin.
         * DateUtil is relying on moment.js. So format pattern of date is referenced here:
         * http://momentjs.com/docs/#/parsing/string-format/
         */

        /**
         * Pattern of string date when use is inserting a date in editor
         */
        PATTERN_INSERTING: "DD-MM-YYYY",

        /**
         * Patten of string date when showing in date as lozenge shape in editor (ADG format)
         * This is overrided by AJS.Meta.get("date.format");
         */
        PATTERN_LOZENGE: "DD MMM YYYY",

        /**
         * Pattern of string date of datetime attribute of <time> tag
         */
        PATTERN_INSIDE_TIME_ELE: "YYYY-MM-DD",

        /**
         * Convert a date object to HTML <time> tag as default, or any custom HTML tag
         * <time> tag have 2 attributes
         * - datetime: have data format as PATTERN_INSIDE_TIME_ELE
         * - contenteditable=false in Chrome, Firefox, Safari. But contenteditable=true in IEs
         * @param {Object} date
         * @param {String} tagName [tagName=<time>]
         * @param {String} className class name of HTML tag
         * @param {String} pattern of date object
         * @returns {String} html string to represent date object
         */
        convertDateIntoHtml: function(date, tagName, className, pattern){
            if (!pattern) {
                pattern = DateUtil.PATTERN_INSERTING;
            }

            if (!className) {
                className = "";
            }

            if (!tagName) {
                tagName = "<time>";
            }

            var $html = $("<div>");
            var $htmlInner = "";
            var dateText = date.format(pattern);

            if (tagName.indexOf("time") >= 0) {
                $htmlInner = $(tagName, {
                    "datetime": date.format(DateUtil.PATTERN_INSIDE_TIME_ELE) //format date in attribute of <time> is always yyyy-MM-dd
                });

                InlineTasksUtil.addContentEditableIfApplicable($htmlInner);

                className && $htmlInner.attr("class", className);

                $htmlInner.text(dateText);
            }
            else {
                $htmlInner = $(tagName);
                className && $htmlInner.attr("class", className);
                $htmlInner.text(dateText);
            }

            $html.append($htmlInner);

            return $html.html();
        },

        /**
         * Insert a date object into editor at current cursor
         * @param date
         * @param tagName
         * @param className
         * @param pattern
         * @param [contentAfter]
         * @returns {*}
         */
        insertDateIntoCurrentCursor: function(date, tagName ,className, pattern, contentAfter){
            var tinymce = require('tinymce');
            var NodeUtils = require('confluence-editor/utils/tinymce-node-utils');
            var ed = tinymce.activeEditor;
            var $today = DateUtil.convertDateIntoHtml(date, tagName, className, pattern);
            var node = $($today, ed.getDoc())[0];
            var newTimeNode = NodeUtils.replaceSelection(node);

            //CONFDEV-22225: add white space between the lozenge and text caret,
            //White space should be "&nbsp;", " " does not work stable when inserting and removing many times
            if (contentAfter) {
                ed.execCommand('mceInsertContent', false, contentAfter, {skip_undo: true});
            }

            // Trigger an analytics event when the date lozenge is added.
            var contextValue = $(ed.selection.getRng(true).startContainer).closest("li[data-inline-task-id]").length  ? 'task' : 'page';
            var data = {context: contextValue, trigger: 'character'};
            AJS.trigger('analyticsEvent', {name: 'confluence-spaces.date.added', data : data});

            return newTimeNode;
        },

        /**
         * Parse a date as string into date object
         * @param dateString data as string
         * @param pattern format string to parse
         * @returns {Date|null} date object here is moment date object, return null in case cannot parse
         */
        parse: function(dateString, pattern) {
            if (!pattern) {
                pattern = DateUtil.PATTERN_INSIDE_TIME_ELE;
            }

            var date = moment($.trim(dateString), pattern, true);
            return date.isValid() ? date : null;
        },

        /**
         * Convert a date object into string base on a specific pattern
         * @param date object
         * @param pattern string date pattern
         * @returns {String} date as string or null
         */
        format: function(date, pattern) {
            if (!pattern) {
                pattern = DateUtil.PATTERN_INSIDE_TIME_ELE;
            }

            return date ? date.format(pattern) : null;
        },
        /**
         * Return a sensible date for a partial date string. The missing fields
         * will be filled with the corresponding fields of today.
         *
         * @param partial a partial date string
         * @param format variants of DD-MM-YYYY, only varies in the order of fields
         * @returns Date valid Date object, or null if the date can't be parsed
         */
        guessPartialDate: function(partial, format) {
            if (partial == null) {
                return null;
            }

            // accept . - or / as separator
            partial = $.trim(partial.replace(/[-\.\/]/g, "-"));

            var parts = partial.split("-");
            var todayParts = moment().format(format).split("-");

            for (var i = 0; i < 3; i++) {
                // fill missing fields with the fields from today
                if (!parts[i]) {
                    parts[i] = todayParts[i];
                }
                // pad zero to single digit
                if (/^\d$/.test(parts[i])) {
                    parts[i] = "0" + parts[i];
                }
            }

            return DateUtil.parse(parts.join("-"), format);
        },

        datepicker: {
            POSITION_ABOVE: true,
            POSITION_BELOW: false,
            POSITION_RIGHT: true,
            POSITION_LEFT: false,

            dropDownStillFitsVertically: function(preferredVerticalPosition, spaceAvailable, heightRequired) {
                if (preferredVerticalPosition === this.POSITION_ABOVE) {
                    return spaceAvailable.above >= heightRequired;
                }
                return spaceAvailable.below >= heightRequired;
            },

            dropDownStillFitsHorizontally: function(preferredHorizontalPosition, spaceAvailable, widthRequired) {
                if (preferredHorizontalPosition === this.POSITION_RIGHT) {
                    return spaceAvailable.right >= widthRequired;
                }
                return spaceAvailable.left >= widthRequired;
            },

            getPreferredHorizontalPosition: function(spaceAvailable, widthRequired) {
                if (spaceAvailable.right >= widthRequired) {
                    return this.POSITION_RIGHT;
                } else if (spaceAvailable.left >= widthRequired) {
                    return this.POSITION_LEFT;
                }

                return spaceAvailable.right > spaceAvailable.left ? this.POSITION_RIGHT : this.POSITION_LEFT;
            },

            getPreferredVerticalPosition: function(spaceAvailable, heightRequired) {
                if (spaceAvailable.below >= heightRequired) {
                    return this.POSITION_BELOW;
                } else if(spaceAvailable.above >= heightRequired) {
                    return this.POSITION_ABOVE;
                }
                // Not enough space - so use biggest space
                return spaceAvailable.below > spaceAvailable.above ? this.POSITION_BELOW : this.POSITION_ABOVE;
            },

            calculateDatepickerPosition: function ($attachTo) {
                var thiz = this;
                return function (popup, targetPosition) {
                    var verticalSpaceAvailable;
                    var horizontalSpaceAvailable;
                    var top;
                    var left;
                    var offset;
                    var gapForArrowY = 16;
                    var gapForArrowX = 0;
                    var ddHeight;
                    var ddWidth;
                    var heightRequired;
                    var widthRequired;
                    var AtlassianEditor = require('confluence-editor/editor/atlassian-editor');
                    var AtlassianEditorContent = require('confluence-editor/editor/atlassian-editor-content');
                    var iframe = AtlassianEditor.Rte.getEditorFrame();


                    if (Position.spaceLeftRight === undefined) {
                        Position.spaceLeftRight = function(containerElement, element) {
                            var elementPos = element.position().left;
                            return {
                                left: elementPos,
                                right: $(containerElement).width() - elementPos
                            };
                        };
                    }

                    verticalSpaceAvailable = Position.spaceAboveBelow(iframe, $attachTo);
                    horizontalSpaceAvailable = Position.spaceLeftRight(iframe, $attachTo);

                    offset = AtlassianEditorContent.offset($attachTo);

                    ddHeight = popup.outerHeight(true);
                    ddWidth = popup.outerWidth(true);
                    heightRequired = ddHeight + gapForArrowY;
                    widthRequired = ddWidth + gapForArrowX;

                    var preferredVerticalPosition = thiz.POSITION_BELOW;
                    var preferredHorizontalPosition = thiz.POSITION_RIGHT;

                    if (preferredVerticalPosition) {
                        if (!thiz.dropDownStillFitsVertically(preferredVerticalPosition, verticalSpaceAvailable, heightRequired)) {
                            preferredVerticalPosition = null;
                        }
                    }

                    if (!preferredVerticalPosition) {
                        preferredVerticalPosition = thiz.getPreferredVerticalPosition(verticalSpaceAvailable, heightRequired);
                    }

                    if (preferredHorizontalPosition) {
                        if (!thiz.dropDownStillFitsHorizontally(preferredHorizontalPosition, horizontalSpaceAvailable, heightRequired)) {
                            preferredHorizontalPosition = null;
                        }
                    }
                    preferredHorizontalPosition = thiz.getPreferredHorizontalPosition(horizontalSpaceAvailable, widthRequired);

                    if (preferredVerticalPosition === thiz.POSITION_ABOVE) {
                        top = offset.top - ddHeight - gapForArrowY;
                    } else {
                        top = offset.top + $attachTo.height() + gapForArrowY;
                    }

                    if (preferredHorizontalPosition === thiz.POSITION_RIGHT) {
                        left = offset.left - gapForArrowX;
                    } else {
                        left = (offset.left + $attachTo.outerWidth()) - ddWidth;
                    }

                    var popupCss = {
                        top: top,
                        left: left
                    };

                    var arrowCssLeft = (preferredHorizontalPosition === thiz.POSITION_RIGHT) ? $attachTo.outerWidth()/2 : ddWidth - $attachTo.outerWidth()/2;

                    // CONFDEV-23097 - If that arrow is outside of the AUI inline dialog due to the autocomplete
                    // line spilling over onto the next line, force it to be on the left of the dialog.
                    if ((arrowCssLeft < 0) || (arrowCssLeft > ddWidth)) {
                        arrowCssLeft = (ddWidth/10);
                    }

                    var arrowCss = {
                        left: arrowCssLeft,
                        top: (preferredVerticalPosition === thiz.POSITION_ABOVE) ? ddHeight : -gapForArrowY/2
                    };

                    return {
                        displayAbove: preferredVerticalPosition,
                        popupCss: popupCss,
                        arrowCss: arrowCss
                    };
                };
            },

            /**
             * Depending whether a date is set, will return different hint text for date picker
             * @param {boolean} isSetDueDate
             * @returns {String}
             */
            getHintTextOfDatePicker: function(isSetDueDate){
                return isSetDueDate
                    ? AJS.I18n.getText("inline-tasks.editor.datepicker.hint.selectduedate")
                    : AJS.I18n.getText("inline-tasks.editor.datepicker.hint.selectdate");
            },

            /**
             * Create a date-picker object corresponding with DOM node
             * @param options has following options
             * - $attachTo: is a jQuery object which hidden input is created automatically and appended before it
             * - $positionTo: is a jQuery object which date-picker will use to re-position to follow
             * - startDate: is a date object will be set when date-picker appears first time
             * - onSelect: is a callback function when a date is selected in date-picker
             * - isSetDueDate: will decide how hint copy appears at top of date-picker
             * @returns {{die: die, show: show, hide: hide, getContainer: getContainer, setDate: setDate, placeDatePicker: placeDatePicker}}
             */
            create: function(options) {
                var $attachTo = options.$attachTo;
                var $positionTo = options.$positionTo;
                var startDate = options.startDate;
                var onSelect = options.onSelect;
                var isSetDueDate = options.isSetDueDate;

                var $datePickerControl = $("<input>").attr("type", "date").addClass("aui-date-picker").css("display", "none");
                var picker = $datePickerControl.inlineTasksDatePicker1({
                    "overrideBrowserDefault" : true,
                    "dateFormat" : "yy-mm-dd",
                    languageCode: AJS.Meta.get("user-locale") && AJS.Meta.get("user-locale").split("_")[0],
                    "position": this.calculateDatepickerPosition($positionTo),
                    "onSelect": function (date) {
                        onSelect(DateUtil.parse(date));
                        killDatePicker();
                    },
                    hint: this.getHintTextOfDatePicker(isSetDueDate)
                });

                var killDatePicker = function(){
                    picker && picker.destroyPolyfill && picker.destroyPolyfill();
                    $datePickerControl && $datePickerControl.remove();
                    //remove unused dialog from DOM tree
                    $(".aui-datepicker-dialog").remove();

                    picker = null;
                    $datePickerControl = null;
                };

                picker.show();
                picker.setDate(DateUtil.format(startDate));

                //for testable
                $(".aui-datepicker-dialog .aui-datepicker-hint").addClass(isSetDueDate ? "set-duedate" : "set-date");

                $attachTo.before($datePickerControl);

                return {
                    die: function() {
                        killDatePicker();
                    },
                    show: function () {
                        picker.show();
                    },
                    hide: function () {
                        picker.hide();
                    },
                    getContainer: function () {
                        return $datePickerControl;
                    },
                    setDate: function (dateText) {
                        var valid = DateUtil.guessPartialDate(dateText, DateUtil.PATTERN_INSERTING);
                        AJS.debug("Set date: " + dateText);
                        valid && picker.setDate(DateUtil.format(valid));
                    },
                    placeDatePicker: function(){
                        var currentHash = AJS.InlineDialog.current;
                        var popup = currentHash && currentHash.popup;
                        popup && popup.refresh();
                    }
                };
            }
        }
    };

    return DateUtil;
});

require('confluence/module-exporter').exportModuleAsGlobal('confluence-inline-tasks/dateutil', 'Confluence.InlineTasks.DateUtil', function (DateUtil) {
    var AJS = require('ajs');
    var _ = require('underscore');
    var Meta = require('confluence/meta');
    var moment = require('moment');

    AJS.toInit(function() {
        "use strict";

        // to format DD-MM-YYYY with the same fields order as mediumFormat
        var toInsertingFormat = function(mediumFormat, locale, separator) {
            // as ADG, viewing date format of US and UK is the same
            // but we want special inserting date format for US
            if (locale && locale.toLowerCase() === "en_us") {
                return "MM" + separator + "DD" + separator + "YYYY";
            }
            var orders = _.uniq(mediumFormat.replace(/[^MDY]/g, "").split(""));
            return orders.join(separator).replace("D", "DD").replace("M", "MM").replace("Y", "YYYY");
        };

        var userLocale = Meta.get("user-locale");
        if (userLocale) {
            moment.lang(userLocale.split("_")[0].toLowerCase());
        }

        var dateFormat = Meta.get("user-date-pattern");
        if (dateFormat) {
            DateUtil.PATTERN_LOZENGE = dateFormat.toUpperCase();
            DateUtil.PATTERN_INSERTING = toInsertingFormat(dateFormat.toUpperCase(), userLocale, "-");
            DateUtil.PATTERN_INSERTING_ALTERNATE = toInsertingFormat(dateFormat.toUpperCase(), userLocale, "/");
        }

    });
});
