define('confluence/ic/view/resolved-comment-list',
    [
        'jquery',
        'underscore',
        'backbone',
        'ajs',
        'confluence/ic/view/resolved-comment-item'
    ],
    function(
        $,
        _,
        Backbone,
        AJS,
        ResolvedCommentView
        ) {

        "use strict";

        var ResolvedCommentListView = Backbone.View.extend({

            events: {
                'click #resolved-dialog-close-button' : 'destroy'
            },

            template: Confluence.Templates.IC.resolvedCommentDialog,

            initialize: function() {
                _.bindAll(this, 'render', 'destroy', '_onKeyDown');
                this._resolvedCommentViews = [];
                this.listenTo(this.collection, 'change:resolveProperties', this.onResolvedChanged);
                this.listenTo(Backbone, "ic:resolvedError", this.displayError);
                this.listenTo(Backbone, "ic:resolvedClearError", this.clearErrors);

                var $dialogHTML = $(this.template());
                this.setElement($dialogHTML);
                this.dialog = AJS.dialog2(this.$el);
                $(document).bind("keydown", this._onKeyDown);
                AJS.trigger("remove-bindings.keyboardshortcuts");
            },

            render: function() {
                var $content = $('<div>');
                var totalResolvedComments = this.collection.getResolvedCount();
                if (totalResolvedComments !== 0) {
                    _.each(this.collection.getResolvedCommentsDesc(), function(comment) {
                        var resolvedCommentView = new ResolvedCommentView({model: comment});
                        this._resolvedCommentViews.push(resolvedCommentView);
                        $content.append(resolvedCommentView.render().$el);
                    }, this);
                } else {
                    this.displayNoCommentMessage();
                }

                this.$('.aui-dialog2-content').append($content);
                this.dialog.show();

                if (this.options.focusCommentId) {
                    this.scrollToComment(this.options.focusCommentId);
                }

                Backbone.trigger("ic:resolved:view", totalResolvedComments);
                // HACK!!! for render JIM asynchronous, rendering resolving dialog
                AJS.trigger("ic-jim-async-supported");
                return this;
            },

            onResolvedChanged: function() {
                if (this.collection.getResolvedCount() === 0) {
                    this.displayNoCommentMessage();
                }
            },

            displayNoCommentMessage: function() {
                AJS.messages.generic(this.$('.aui-dialog2-content'), {
                    body: AJS.I18n.getText('inline.comments.resolved.dialog.empty'),
                    closeable: false
                });
            },

            displayError: function(error) {
                this.clearErrors();
                AJS.messages.error(this.$('#ic-resolved-comment-dialog-error'), {
                    body: error,
                    closeable: false
                });
            },

            clearErrors: function() {
                this.$('#ic-resolved-comment-dialog-error').empty();
            },

            scrollToComment : function(commentId) {
                var $commentElement = this.$(AJS.format("[data-comment-id={0}]", commentId));
                if ($commentElement.length) {
                    $commentElement.addClass('focused');
                    return this.$('.aui-dialog2-content').animate({
                        scrollTop: $commentElement.parent().position().top
                    }, 100);
                }

                var deferred = new $.Deferred();
                return deferred.reject();
            },

            _onKeyDown: function(e) {
                //check $(e.target).is(:input) here and ignore e.preventDefault() for typing text
                if (this.$el && this.$el.is(':visible') && e.keyCode === AJS.keyCode.ESCAPE) {
                    this.destroy();
                }
            },

            destroy: function() {
                AJS.trigger("add-bindings.keyboardshortcuts");
                $(document).unbind("keydown", this._onKeyDown);
                this.dialog.hide();
                this.dialog.remove();
                _.each(this._resolvedCommentViews, function(view) {
                    view.destroy();
                });
                this._resolvedCommentViews = null;
                this.remove();
            }
        });

        return ResolvedCommentListView;
    });
