define('confluence/ic/view/reply-list',
    [
        'jquery',
        'ajs',
        'underscore',
        'backbone',
        'confluence/ic/view/display-reply',
        'confluence/ic/util/utils'
    ],
    function(
        $,
        AJS,
        _,
        Backbone,
        DisplayReplyView,
        Utils
    ) {

    "use strict";

    var ReplyListView = Backbone.View.extend({
        tagName: "div",

        className: "ic-replies",

        events: {
            "click #ic-showmore-container": "forceShowAllReplies"
        },

        initialize: function(options) {
            if (!options.commentModel) {
                throw new Error('Inline Comment Model should be passed as a parameter when init ReplyListView', 'ReplyListView');
            }
            this.replyViews = [];
            this.collection = options.commentModel.replies;
            this.source = options.source;

            // both reply list and resolved comment reply list fetch the replies collection causing both views to
            // re-render. This ensures that we only render the view once on fetch and then listen for add/remove events only
            this.listenTo(this.collection, "add remove", this.render);
            if (this.collection.isFetched !== true) {
                this.listenToOnce(this.collection, "sync", this.fetchComplete);
                this.collection.fetch({reset: true});
            }

            this.expandReplies = false;
            this.COMMENT_NUMBER_TO_COLLAPSE = 4;
        },

        fetchComplete: function() {
            this.collection.isFetched = true;
            this.render();
            //CONFDEV-30990
            Utils.resizeContentAfterLoadingImages(this.$el);
        },

        render: function(newReply) {
            if (this.collection.isFetched !== true) {
                return this;
            }

            // newReply is defined when render() is triggered as a result of Backbone collection.add()
            if (newReply && this.collection.last() === newReply) {
                this.appendReply(newReply);
                return;
            }
            this.cleanUpReplyViews();

            // clean the element before rendering it
            this.$el.empty();

            if (this.collection.length > this.COMMENT_NUMBER_TO_COLLAPSE
                    && this.expandReplies === false
                    && this.source !== 'permalink'
                    && this.options.commentModel && !this.options.commentModel.isResolved()) {
                this.renderCollapsibleReplies();
            } else {
                this.renderFullReplies();
            }

            this.trigger("ic:reply-rendered", this.options.commentModel);
            this._hightlightReply();

            // HACK!!! for render JIM asynchronous, for rendering reply
            AJS.trigger("ic-jim-async-supported");

            return this;
        },

        renderFullReplies : function() {
            this.collection.each(function(reply) {
                this.appendReply(reply);
            }, this);
        },

        forceShowAllReplies: function() {
            this.expandReplies = true;
            this.render();
        },

        renderCollapsibleReplies: function() {
            this.renderExpendButton();
            var reply = this.collection.last();
            this.appendReply(reply);
        },

        appendReply : function(reply) {
            var replyView = this.createReplyView(reply);
            this.replyViews.push(replyView);
            this.$el.append(replyView.render().el);
        },

        renderExpendButton : function() {
            this.$el.append(Confluence.Templates.IC.showmoreReply({totalReplies : this.collection.length - 1}));
        },

        createReplyView : function(reply) {
            return new DisplayReplyView({
                model: reply,
                resolved: this.options.commentModel.isResolved() // this option will decide to show ic-actions dom or not
            });
        },

        _hightlightReply: function() {
            if (this.source !== 'permalink') return;

            var query = window.location.search;
            var match = query.match(/(focusedCommentId|replyToComment)=(\d+)/); // match[2] will hold the id if found
            if (match == null) {
                return;
            }
            var replyId = parseInt(match[2], 10);
            if (replyId && replyId !==0) {
                this.$("[data-comment-id=" + replyId + "]").addClass('ic-comment-reply-highlight');
            }
        },

        cleanUpReplyViews: function() {
            var i;
            for (i = 0; i < this.replyViews.length; i++) {
                this.replyViews[i].destroy();
                this.replyViews[i] = null;
            }
            this.replyViews = [];
        },

        destroy: function() {
            this.cleanUpReplyViews();
            this.remove();
        }
    });

    return ReplyListView;
});
