define('confluence/ic/view/display-reply',
    [
        'jquery',
        'ajs',
        'underscore',
        'backbone',
        'confluence/ic/util/utils',
        'confluence/ic/view/create-comment',
        'confluence/ic/view/likes'
    ],
    function(
        $,
        AJS,
        _,
        Backbone,
        Utils,
        CreateCommentView,
        LikesView
    ) {

    "use strict";

    var DisplayReplyView = Backbone.View.extend({
        tagName: "div",

        className: "ic-display-reply",

        template: Confluence.Templates.IC.renderReply,

        events: {
            // also need to bind an event for canceling and saving the comment
            "click .ic-action-edit-reply": "editReply",
            "click .ic-action-delete-reply": "confirmDelete"
        },

        initialize: function() {
            _.bindAll(this, "onEditCallback");
            this.$el.attr('data-comment-id', this.model.get("id"));
        },

        render: function() {
            var data = _.extend(this.model.toJSON(), {
                resolved: this.options.resolved,
                darkFeatures: Utils.getDarkFeatures()
            });
            if (AJS.Meta.get("render-mode") === 'READ_ONLY') {
                data.hasEditPermission = false;
                data.hasDeletePermission = false;
            }
            var html = this.template(data);
            this.$el.html(html);
            this._renderLikes();
            return this;
        },

        _renderLikes: function() {
            if (this.likesView) {
                this.likesView.remove();
            }
            this.likesView = new LikesView({contentId: this.model.id}).render();
            if(this.likesView.$el.is(':empty')) {
                this.$('.ic-action-like-reply').remove(); // remove the redundant trailing dot from the container view
            } else {
                this.$('.ic-action-like-reply').html(this.likesView.el);
            }
        },

        editReply: function(e) {
            var that = this;
            if (Utils.confirmProcess()) {
                var promise = Utils.getEditorFormat(this.model.get("id"));
                promise.done(function(editBody) {
                    that.editView = new CreateCommentView({
                        model: that.model,
                        type: "edit",
                        selection: {},
                        onFinished: that.onEditCallback,
                        commentText: editBody
                    });
                    that.$el.html(that.editView.$el);
                    that.editView.render();
                }).fail(function() {
                    Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.server.error.edit.reply'));
                });
            } else {
                // prevent Edit button has a green outline after cancel confirm dialog
                e && e.currentTarget.blur();
            }
        },

        onEditCallback: function() {
            this.editView && this.editView.destroy();
            this.editView = null;
            this.render();
        },

        deleteFail: function(model, response) {
            switch (response.status) {
                case 401:
                    Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.delete.permission'));
                    break;
                case 405:
                    if (response.responseText) {
                        var responseJson = JSON.parse(response.responseText);
                        if (responseJson.reason === "READ_ONLY") {
                            Backbone.trigger("ic:error", AJS.I18n.getText('read.only.mode.default.error.short.message'));
                            break;
                        }
                    }
                default:
                    Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.server.error'));
            }
        },

        confirmDelete: function(e) {
            var answer = window.confirm(AJS.I18n.getText('inline.comments.delete.reply.confirmation.message'));
            if (answer === true) {
                this.deleteReply(e);
            }
        },

        deleteReply: function(e) {
            // disable the delete button, so we can only click it once
            $(e.target).attr("aria-disabled", true).prop("disabled", true);
            this.model.destroy({
                wait: true,
                error: this.deleteFail,
                success: function() {
                    Backbone.trigger("ic:reply:delete");
                }
            });
        },

        destroy: function() {
            this.likesView && this.likesView.remove();
            this.editView && this.editView.destroy();
            this.remove();
        }
    });

    return DisplayReplyView;
});
