define('confluence/ic/util/utils',
    [
        'jquery',
        'underscore',
        'ajs',
        'backbone',
        'exports'
    ],
    function(
        $,
        _,
        AJS,
        Backbone,
        exports
    ) {

    "use strict";

    var darkFeatures = {
            INLINE_COMMENTS: AJS.DarkFeatures.isEnabled('confluence-inline-comments'),
            RESOLVED_INLINE_COMMENTS: AJS.DarkFeatures.isEnabled('confluence-inline-comments-resolved'),
            RICH_TEXT_EDITOR: AJS.DarkFeatures.isEnabled('confluence-inline-comments-rich-editor'),
            DANGLING_COMMENT: AJS.DarkFeatures.isEnabled('confluence-inline-comments-dangling-comment')
        },

        unsupportedRtePlugins = [
            'dateautocomplete',
            'confluencemacrobrowser',
            'propertypanel',
            'jiraconnector',
            'dfe'
        ],

        supportedRtePlugins = [
            'autoresize',
            'confluenceleavecomment'
        ],

        unsupportedMacros = [
            'code'
        ]
        ;
    var $pageContainer, $splitterContent, _isDocTheme;

    function getDarkFeatures() {
        return _.clone(darkFeatures);
    }

    /*
     * Return array of Rich Text Editor plugins that Inline Comment does not support
     */
    function getUnsupportedRtePlugins() {
        return _.clone(unsupportedRtePlugins);
    }

    function getSupportedRtePlugins() {
        return _.clone(supportedRtePlugins);
    }

    /*
     * See if the current selection overlaps an existing comment, if it does, return the comment, if it does not
     * return undefined
     */
    function overlappedSelection(selectionObject, commentCollection) {
        var comment;

        if (!selectionObject || !commentCollection) {
            return;
        }

        var $overlappingMarkers = $(selectionObject.containingElement);
        if (!$overlappingMarkers.is('.inline-comment-marker.valid')) {
            $overlappingMarkers = $('<div/>').append(selectionObject.html).find('.inline-comment-marker.valid');
        }

        if ($overlappingMarkers.length > 0 && _isContainMarkerHasText($overlappingMarkers)) {
            var overlapMarkerRef = $overlappingMarkers.first().data('ref');
            comment = commentCollection.findWhere({ markerRef: overlapMarkerRef });
            return comment;
        }
    }

    //In IE, when we highlight a text closet the inline comment wrapper. It will contain the empty inline comment element
    function _isContainMarkerHasText($overlappingMarkers) {
        return  $overlappingMarkers.filter(function() {
                return $(this).text().length > 0;
            }).length > 0;
    }

    function focusedCommentUrl(commentId) {
        return AJS.contextPath() + '/pages/viewpage.action?pageId=' + AJS.Meta.get('latest-page-id') +
            '&focusedCommentId=' + commentId + '#comment-' + commentId;
    }

    /*
     * Allow permalink to inline comments. If the focusedCommentId present in the url query string matches an
     * inline comment id, display the comment
     */
    function showFocusedComment(collection, query, ResolvedCommentListView) {
        var match = query.match(/(focusedCommentId|replyToComment)=(\d+)/); // match[2] will hold the id if found
        if (match == null) {
            return;
        }

        var isReplyComment = match[1] === 'replyToComment'; // in-case user click on reply link, Editor should be activated
        var commentId = parseInt(match[2], 10);
        var model = collection.findWhere({ id: commentId });

        if (model != null) {
            _showPermalinkedComment(collection, model, ResolvedCommentListView, isReplyComment);
        } else {
            $.ajax({
                url: AJS.contextPath() + "/rest/inlinecomments/1.0/comments/replies/" + commentId + "/commentId"
            }).done(function(id) {
                model = collection.findWhere({ id: id });
                _showPermalinkedComment(collection, model, ResolvedCommentListView, isReplyComment);
            });
        }
    }

    function _showPermalinkedComment(collection, model, ResolvedCommentListView, isReplyComment) {
        if (model != null) {
            if (model.isResolved()) {
                new ResolvedCommentListView({
                    collection: collection,
                    focusCommentId: model.get("id")
                }).render();
            } else {
                if (!model.isDangling()) { // only if comment is not dangling
                    Backbone.trigger("ic:view", model, 'permalink', {isReplyComment: isReplyComment});
                }
            }
        }
    }


    function getAuthorAvatarUrl() {
        if(AJS.Meta.get('current-user-avatar-url')) {
            return AJS.contextPath() + AJS.Meta.get('current-user-avatar-url');
        }
        return AJS.Meta.get('static-resource-url-prefix') + "/images/icons/profilepics/anonymous.png";
    }

    function getAuthorDisplayName() {
        return AJS.Meta.get('user-display-name') || AJS.Meta.get('current-user-fullname');
    }

    /*
     * Internal implementation for moveCaretToEnd
     */
    function _moveCaretToEnd(fieldEl) {
        if (typeof fieldEl.selectionStart === "number") {
            var inputLength = fieldEl.value.length;
            fieldEl.selectionStart = inputLength;
            fieldEl.selectionEnd = inputLength;
        } else if (typeof fieldEl.createTextRange != null) {
            fieldEl.focus();
            var range = fieldEl.createTextRange();
            range.collapse(false);
            range.select();
        }
    }

    /*
     * This helper is used to Edit a comment. When editing a textarea, it will place the cursor at the end of the
     * existing text. It must also get called after a timeout because Chrome cannot handle the cursor set synchronously
     */
    function moveCaretToEnd(fieldEl) {
        _moveCaretToEnd(fieldEl);
        window.setTimeout(function() {
            _moveCaretToEnd(fieldEl);
        }, 1);
    }

    function animateSidebar(sidebar, isSlideIn) {
        var endAnimationEventName = 'webkitAnimationEnd oanimationend msAnimationEnd animationend';
        var $wikiContent = sidebar.$wikiContent;
        var deferred = $.Deferred();

        if(isSlideIn) {
            var activeHighlights = $('.inline-comment-marker.active');

            $wikiContent.addClass('ic-fade-in-animation');
            $wikiContent.one(endAnimationEventName, function() {
                $(this).removeClass('ic-fade-in-animation');
            });

            sidebar.$el.addClass('ic-slide-in');
            sidebar.$el.one(endAnimationEventName, function() {
                $(this).removeClass('ic-slide-in');
                activeHighlights.addClass('ic-highlight-pulse');
                activeHighlights.one(endAnimationEventName, function() {
                    activeHighlights.removeClass('ic-highlight-pulse');
                })
            });
            deferred.resolve();
        } else {
            $wikiContent.bind(endAnimationEventName, function(e) {
                if(e.originalEvent.animationName == 'ic-fade-out-animation') {
                    sidebar.$wikiContent.removeClass('ic-fade-out-animation');
                    sidebar.$wikiContent.css('opacity', '0.5');
                } else if (e.originalEvent.animationName == 'ic-fade-in-animation') {
                    sidebar.$wikiContent.css('opacity', '1');
                    sidebar.$wikiContent.removeClass('ic-fade-in-animation');
                    sidebar.$wikiContent.unbind(endAnimationEventName);
                }
            });
            $wikiContent.addClass('ic-fade-out-animation');

            sidebar.$el.addClass('ic-slide-out');

            /* CONF-37937
                Need to on off the event manually instead of using $.fn.one() because if these conditions are
                met then the handler code will be executed twice:
                - Chrome version is >= 43.0.x
                - handler is bound using $.fn.one()
                - webkitAnimationEnd animationend are listened using the same hander
                - event propagation isn't stopped

                The cause is that animationend event is later on transformed to webkitAnimationEnd (and fired again) which causes
                the handler code to be executed twice, this doesn't happen in Chrome < 43.0.x

                In this specific case, the event was bubbling up from "form.aui" of the editor
             */
            var postAnimateHandler = function() {
                sidebar.$el.removeClass('ic-slide-out');
                sidebar._emptySidebar();
                getPageContainer().css("padding-right", "0");
                $wikiContent.addClass('ic-fade-in-animation');
                $wikiContent.css("position", "static");
                deferred.resolve();
                sidebar.$el.off(endAnimationEventName, postAnimateHandler);
            };
            sidebar.$el.on(endAnimationEventName, postAnimateHandler);
        }

        return deferred.promise();
    }

    function getInlineLinks(highlight) {
        var $links  = $(highlight).closest('a');
        if(!$links.length) {
            //when edit a current highlight and add a link to highlight. Link will be a child of highlight
            $links = $(highlight).find('a');
        }

        return $links;
    }

    function removeInlineLinksDialog(highlights) {
        highlights.each(function() {
            var $links = getInlineLinks(this);
            $links.length && $links.off('mousemove');
        })
    }

    /*
     * Check browse support animation css
     */
    function isAnimationSupported() {
        return document.body.style.animation !== undefined ||
               document.body.style.webkitAnimation !== undefined ;
    }

    /*
     * Make the highlight content visible
     * Current code handle for expand macro
     */
    function showHighlightContent($highlight) {
        //handle for expand macro
        var $expands = $highlight.parents('.expand-content.expand-hidden');
        $expands.each(function(expand) {
            $(this).siblings('.expand-control').click();
        });
    }

    /**
     * Show confirm dialog if user is editing
     * @returns boolean
     */
    function confirmProcess(checkSidebarOnly) {
        var openingEditor = AJS.Rte && AJS.Rte.getEditor();

        if (checkSidebarOnly === true) {
            if (hasEditorInSidebar() && openingEditor && openingEditor.isDirty()) {
                return confirm(AJS.I18n.getText("inline.comments.editor.lost"));
            }
        } else {
            if (openingEditor && openingEditor.isDirty() && Confluence.Editor && !Confluence.Editor.getContentType) {
                return confirm(AJS.I18n.getText("inline.comments.editor.lost"));
            }
        }

        return true;
    }

    function hasEditorInSidebar() {
        return !!$('.ic-sidebar #wysiwygTextarea_ifr').length;
    }

    /**
     * Check confluence enable/disable keyboard shortcuts
     * @returns boolean
     */
    function isKeyboardShortcutsEnable() {
        return AJS.Meta.get('use-keyboard-shortcuts') &&
            Confluence.KeyboardShortcuts && Confluence.KeyboardShortcuts.enabled;
    }

    function addSidebarHeadingTooltip(view) {
        view.$('button.ic-action-hide').tooltip({gravity: 'se'});
        view.$('#ic-nav-next').tooltip({gravity: 's'});
        view.$('#ic-nav-previous').tooltip({gravity: 's'});
    }

    /**
     * Calculate height of IC sidebar in order to resize the page container if needed
     * @param $container container of IC
     */
    function recalculateContentHeight($container) {
        if (isDocTheme()) {
            // content is auto-flow in doctheme, no need to adjust the height of #content
            $container.css('padding-bottom','20px');
            return;
        }

        var commentThreadHeight = $container.height();
        var threadYPosTop = $container.offset().top;
        var threadYPosBottom = threadYPosTop + commentThreadHeight;

        // threadYPosBottom is the absolute bottom position with relation to viewport. Need to subtract out the
        // space from the parent container to adjust everything accordingly
        var spaceFromParentContainer = this.getPageContainer().offset().top;
        var newHeight = threadYPosBottom - spaceFromParentContainer;
        // if the bottom of the comment thread will overflow the footer, expand the height of the page
        this.getPageContainer().css({ "min-height": newHeight + "px" });
    }

    //CONFDEV-30990: recalculateContentHeight after all images in IC had been loaded into browser
    /**
     * Resize the page container after finishing loading all images inside an IC
     * @param $container container of IC
     */
    function resizeContentAfterLoadingImages($container) {
        $container = $container.closest(".ic-display-comment-view");
        var thiz = this;
        var $images = $(".confluence-embedded-image, .confluence-embedded-file img", $container);
        var totalImages = $images.length;
        var loadedImages = 0;
        if (totalImages > 0) {
            $images.off("load").one("load", function () {
                if (++loadedImages === totalImages) {
                    thiz.recalculateContentHeight($container);
                }
            }).each(function () {
                //fix for Chrome: if image is cached, trigger load event manually
                if (this.complete) {
                    $(this).load();
                }
            });
        };
    }

    function isDocTheme() {
        if (_isDocTheme === undefined) {
            _isDocTheme = !!getSplitterContent().length;
        }
        return _isDocTheme;
    }

    function getPageContainer() {
        if ($pageContainer === undefined) {
            $pageContainer = $('#content');
        }
        return $pageContainer;
    }

    function getSplitterContent() {
        if ($splitterContent === undefined) {
            $splitterContent = $('#splitter-content');
        }
        return $splitterContent;
    }

    function scrollToCommentAfterToggleSideBar(heightToTopOffset, activeHighlight) {
        if (activeHighlight && !activeHighlight.is(":visible")) {
            return;
        }
        if (heightToTopOffset) {
            if (this.isDocTheme()) {
                this.getSplitterContent().scrollTop(activeHighlight.offset().top - heightToTopOffset);
            } else {
                window.scrollTo(0, activeHighlight.offset().top - heightToTopOffset);
            }
        }
    }

    function getEditorFormat(contentID) {
        var dfd = $.Deferred();
        $.ajax({
            url: AJS.contextPath() + "/rest/api/content/" + contentID + "?expand=body.editor",
            type: "GET",
            dataType: "json",
            contentType: "application/json; charset=utf-8"
        }).then(function(data) {
            var editBody = data.body.editor.value;
            dfd.resolve(editBody);
        }).fail(function (jqXHR, textStatus, errorThrown) {
            dfd.reject(jqXHR, textStatus, errorThrown);
        });
        return dfd.promise();
    }

    function containUnsupportedMacro($searchContent) {
        var $closestMacroContainer = $searchContent.closest('.conf-macro');
        return ($closestMacroContainer.data('hasbody') === false || $searchContent.find('.conf-macro[data-hasbody="false"]').length > 0)
                || unsupportedMacros.indexOf($closestMacroContainer.data('macro-name')) != -1;
    }

    function containInternalLink($searchContent) {
        return $searchContent.closest(".user-mention").length > 0
            || $searchContent.find(".user-mention").length > 0;
    }

    function containUserMetion($searchContent) {
        return $searchContent.closest("a[href^='/']:not('.user-mention')").length > 0
            || $searchContent.find("a[href^='/']:not('.user-mention')").length > 0;
    }

    function getCurrentUserInfo() {

        var userProfilePicture = {
            isDefault: true,
            path: AJS.Meta.get("static-resource-url-prefix") + "/images/icons/profilepics/default.png"
        };
        if (AJS.Meta.get("current-user-avatar-url") !== "/images/icons/profilepics/default.png") {
            userProfilePicture = {
                isDefault: false,
                path: AJS.contextPath() + AJS.Meta.get("current-user-avatar-url")
            }
        }
        var username = AJS.Meta.get("remote-user");
        return {
            userName: username == "" ? null : username,
            displayName: AJS.Meta.get("current-user-fullname"),
            profilePicture: userProfilePicture
        };
    }

    exports.overlappedSelection = overlappedSelection;
    exports.focusedCommentUrl = focusedCommentUrl;
    exports.showFocusedComment = showFocusedComment;
    exports.getAuthorAvatarUrl = getAuthorAvatarUrl;
    exports.moveCaretToEnd = moveCaretToEnd;
    exports.animateSidebar = animateSidebar;
    exports.getDarkFeatures = getDarkFeatures;
    exports.getInlineLinks = getInlineLinks;
    exports.removeInlineLinksDialog = removeInlineLinksDialog;
    exports.isAnimationSupported = isAnimationSupported;
    exports.showHighlightContent = showHighlightContent;
    exports.getUnsupportedRtePlugins = getUnsupportedRtePlugins;
    exports.getSupportedRtePlugins = getSupportedRtePlugins;
    exports.confirmProcess = confirmProcess;
    exports.getAuthorDisplayName = getAuthorDisplayName;
    exports.isKeyboardShortcutsEnable = isKeyboardShortcutsEnable;
    exports.addSidebarHeadingTooltip = addSidebarHeadingTooltip;
    exports.hasEditorInSidebar = hasEditorInSidebar;
    exports.recalculateContentHeight = recalculateContentHeight;
    exports.resizeContentAfterLoadingImages = resizeContentAfterLoadingImages;
    exports.isDocTheme = isDocTheme;
    exports.getPageContainer = getPageContainer;
    exports.getSplitterContent = getSplitterContent;
    exports.scrollToCommentAfterToggleSideBar = scrollToCommentAfterToggleSideBar;
    exports.getEditorFormat = getEditorFormat;
    exports.containUnsupportedMacro = containUnsupportedMacro;
    exports.containInternalLink = containInternalLink;
    exports.containUserMetion = containUserMetion;
    exports.getCurrentUserInfo = getCurrentUserInfo;
});