define('confluence/ic/editor/editor',
    [
        'jquery',
        'ajs',
        'confluence/ic/util/utils',
        'exports'
    ],
    function(
        $,
        AJS,
        Utils,
        exports
        ) {

        "use strict";

        function preActivate() {
            if (!AJS.Confluence.EditorLoader.resourcesLoaded()) {
                this.$form.find('.loading-container').spin('small');
            }
        }

        function preInitialise() {
            // Make sure the content-type is set to "comment" when activating a quick comment, see CONFDEV-10526 for more detail
            AJS.Meta.set('content-type', 'comment');
            AJS.Meta.set('use-inline-tasks', 'false'); // disable Inline Task
            AJS.Meta.set('min-editor-height', 80);
        }

        function postInitialise(options) {
            // Hide the loading
            options.$form.find('.loading-container').hide();

            //This elements is using for both Blogpost and Page, for blogpost, text will be Publish, that's why we have to re-set the text to make sure it works correctly. The same as title.
            options.$form.find('#rte-button-publish').text(AJS.I18n.getText('inline.comments.sidebar.save')).removeAttr('title');

            options.$form.find("#toolbar-hints-draft-status").hide();

            if (this.hideCancelButton) {
                options.$form.find("#rte-button-cancel").hide();
            }

            options.editor.onPostProcess.add(resizeEditor("onPostProcess", options.editor));

            // CONFDEV-35279 Confluence.Editor.getContentType is introduced in core after enhancing Quick Edit
            if (Confluence.Editor && !Confluence.Editor.getContentType) {
                // Once Inline Comment Editor is activated, quick edit page will be disabled, It similar to page comment
                AJS.Confluence.QuickEdit.QuickEditPage.disable();
            }

            //CONFDEV-36296 Remove all existing save/cancel handlers
            Confluence.Editor.removeAllCancelHandlers();
            Confluence.Editor.removeAllSaveHandlers();
        }

        /**
         * Setup RTE in sidebar
         * @param option.container {object} The container containing the necessary structure to activate the editor within.
         * @param option.form {object} The target editor form.
         * @param option.saveHandler {function} Save handler.
         * @param option.cancelHandler {function} Cancel handler.
         * @param option.fetchContent {function} Deferred object that will return editor's content
         * @param option.postDeactivate {function} This function will be executed after an RTE is deactivated completely
         *
         * @returns promise tracking editor activation
         */
        function init(option) {
            return AJS.Confluence.QuickEdit.activateEditor({
                preActivate : preActivate, // Gets executed before the call to fetch resources.
                preInitialise: preInitialise, // Gets executed after the resources are fetched but before the editor gets loaded.
                postInitialise: $.proxy(postInitialise, option), // Gets executed after editor has loaded.
                toolbar: false,

                $container : option.container,
                $form : option.form,
                saveHandler: option.saveHandler,
                cancelHandler: option.cancelHandler,
                fetchContent: option.fetchContent(),
                closeAnyExistingEditor: true,
                postDeactivate: option.postDeactivate,
                plugins: Utils.getSupportedRtePlugins(),
                excludePlugins: Utils.getUnsupportedRtePlugins()
            });
        }

        /**
         * Remove editor's dom and relative things.
         * @returns Deferred object
         */
        function remove() {
            if (AJS.Rte && AJS.Rte.getEditor()) {
                return AJS.Confluence.QuickEdit.deactivateEditor();
            } else {
                return $.Deferred().resolve();
            }
        }

        /**
         * Get content from current Editor
         * @returns HTML
         */
        function getContent() {
            return AJS.Rte.Content.getHtml();
        }

        function setEditorBusy(isBusy) {
            //use new spinner
            if (isBusy) {
                $('.ic-sidebar #rte-spinner').spin();
            } else {
                $('.ic-sidebar #rte-spinner').spinStop();
            }
            // set disabled attribute to prevent clicking many times on save button
            $('#rte-button-publish').prop("disabled", isBusy);
        }

        /**
         * Form a handler and remove itself from the dispatcher
         * @param event event name which handler will be added
         * @param editor editor instance
         * @returns {Function} the handler
         */
        function resizeEditor(event, editor) {
            var handler = function () {
                tinymce.DOM.setStyle(tinymce.DOM.get(editor.id + '_ifr'), 'height', '80px');
                //manually run Auto Resize
                editor.execCommand('mceAutoResize', {forceExec: true});
                //remove handler from dispatcher list
                editor[event].remove(handler);
            };
            return handler;
        }

        /**
         * Base on tinymce version to check if editor is empty
         * @param {string} editorContent
         * @returns {boolean}
         */
        function isEmptyContent(editorContent) {
            // TODO: cleanup the condition check once tiny4 is on stable state (remove DF frontend.editor.v4)
            if (AJS.DarkFeatures.isEnabled('frontend.editor.v4')) {
                return editorContent === '';
            } else {
                return editorContent === '<p><br /></p>';
            }
        }

        exports.init = init;
        exports.remove = remove;
        exports.getContent = getContent;
        exports.setEditorBusy = setEditorBusy;
        exports.isEmptyContent = isEmptyContent;
    });
