define('confluence/ic/view/resolved-comment-item',
    [
        'jquery',
        'underscore',
        'ajs',
        'backbone',
        'confluence/ic/util/utils',
        'confluence/ic/view/reply-list',
        'confluence/ic/view/likes'
    ],
    function(
        $,
        _,
        AJS,
        Backbone,
        Utils,
        ReplyListView,
        LikesView
    ) {

        "use strict";

        var ResolvedCommentView = Backbone.View.extend({
            tagName: "div",

            className: "ic-display-comment-view",

            template: Confluence.Templates.IC.comment,

            events: {
                'click .ic-action-unresolve': 'unresolveComment',
                'click .ic-action-show-reply': 'showReply'
            },

            initialize: function() {
                _.bindAll(this, 'onUnresolveComplete', 'destroy');

                this.replyListView = new ReplyListView({
                   commentModel: this.model
                });
                this.listenTo(this.replyListView, "ic:reply-rendered", function() {
                    this.renderNumberOfReplies();
                });
            },

            render: function() {
//                console.log(this.model.replies.length);
//                alert(this.model.replies.length);
                var html = this.template(
                    _.extend(this.model.toJSON(), {
                        showMenu: false,
                        isDangling: this.model.isDangling(),
                        darkFeatures: Utils.getDarkFeatures(),
                        hasReply: this.model.replies.length > 0
                    })
                );
                this.$el.html(html);
                if (this.likesView) {
                    this.likesView.remove();
                }
                this.likesView = new LikesView({contentId: this.model.id, showLikeButton: false}).render();

                this.$('.ic-action-like').html(this.likesView.el);
                this.$el.append(this.replyListView.$el.hide());
                this.replyListView.render();

                return this;
            },

            unresolveComment: function() {
                Backbone.trigger("ic:resolvedClearError");
                this.model.resolve(false, {
                    wait: true,
                    success: this.onUnresolveComplete,
                    error: this.onUnresolveFail
                });
            },

            onUnresolveComplete: function() {
                Backbone.trigger("ic:unresolved");
                this.$el.slideUp(this.destroy);
            },

            onUnresolveFail: function(model, response) {
                if (response.status === 401) {
                    Backbone.trigger("ic:resolvedError", AJS.I18n.getText('inline.comments.resolve.permission'));
                } else {
                    Backbone.trigger("ic:resolvedError", AJS.I18n.getText('inline.comments.server.error'));
                }
            },

            showReply: function() {
                this.$('.ic-action-show-reply').parent().remove(); // remove show reply button
                this.$('.ic-replies').slideDown();
            },

            renderNumberOfReplies: function() {
                var numOfReplies = this.model.replies.length;
                // only show replies toggle if comment has replies
                var $showReplyButton = this.$('.ic-action-show-reply');
                if (numOfReplies > 0) {
                    var replyButtonText;
                    if (numOfReplies === 1) {
                        replyButtonText = AJS.I18n.getText('inline.comments.resolved.menu.show.reply', 1);
                    } else {
                        replyButtonText = AJS.I18n.getText('inline.comments.resolved.menu.show.replies', numOfReplies);
                    }

                    $showReplyButton.text(replyButtonText);
                    $showReplyButton.parent().removeClass("hidden");
                } else {
                    $showReplyButton.parent().remove();
                }
            },

            destroy: function() {
                this.replyListView.destroy();
                this.remove();
            }
        });

        return ResolvedCommentView;
    });
