define('confluence/ic/model/comment',
    [
        'jquery',
        'underscore',
        'backbone',
        'ajs',
        'confluence/ic/util/utils',
        'confluence/ic/model/reply-collection'
    ],
    function(
        $,
        _,
        Backbone,
        AJS,
        Utils,
        ReplyCollection
    ) {

    "use strict";

    // Model for an Inline Comment. Consists of some default values that are static on
    // every page load. A urlRoot to override the Collection's url on model.save(), model.destroy(). A validate
    // function that currently doesn't do much, but we can expand in the future.
    var Comment = Backbone.Model.extend({

        defaults: {
            authorDisplayName: Utils.getAuthorDisplayName(),
            authorUserName: AJS.Meta.get('remote-user'),
            authorAvatarUrl: AJS.contextPath() + AJS.Meta.get('current-user-avatar-url'),
            body: "",
            originalSelection: "",
            containerId: AJS.Meta.get('latest-page-id'),
            parentCommentId: "0", //TODO: remove this!!! we don't need the parent-child relationship between Comment models because we have a separate model for Reply now
            // This should be ajs-confluence-request-time. Won't be able to Mock this in a test :( //
            lastFetchTime: $('meta[name=confluence-request-time]').attr('content'),
            hasDeletePermission: true,
            hasEditPermission: true,
            hasResolvePermission: true,
            resolveProperties: {
                resolved: false,
                resolvedTime: 0
            },
            serializedHighlights: "",
            deleted: false
        },

        // Need to use this url for model.save() and model.destroy() because the collection url
        // does not work due to query parameters present
        urlRoot: AJS.contextPath() + "/rest/inlinecomments/1.0/comments",

        initialize: function (attr)
        {
            attr = attr || {};
            this._setHighlights(attr.markerRef);

            var parent = this;
            this.replies = new ReplyCollection();
            // It's a good practice to overwrite the collection's url here
            // because we can reuse the model's url and know which model the collection is nested in
            // On the other hand, if we bind a specific mode's url in the Collection, it's impossible to fetch the collection
            // if it also belongs to another model (i.e. User)
            // for more complicated relational models, we might need this add-on http://backbonerelational.org/
            this.replies.url = function () {
                return parent.url() + '/replies';
            };

            if (Utils.getDarkFeatures().DANGLING_COMMENT && AJS.Meta.get('page-id') === AJS.Meta.get('latest-page-id')) {
                if (this.isDangling() && !this.isResolved()) {
                    this.resolve(true, {
                        wait: true,
                        dangling: true,
                        success: _.bind(function () {
                            Backbone.trigger("ic:open:dangled", this);
                        }, this),
                        error: _.bind(function() {
                            Backbone.trigger("ic:resolve:dangled:failed", this);
                        }, this)
                    });
                }
            }
        },

        // will abort server request is this fails. If validate returns a "truthy" value, it fails the validation
        // and aborts the server request
        validate: function () {
            if (!this.get("body")) {
                return true;
            }
        },

        resolve: function(resolved, options) {
            options = options || {};
            this.save({}, _.extend(options, {
                    url: this.urlRoot + "/" + this.get("id") + "/resolve/" + resolved + "/dangling/" + (options.dangling === true)
                })
            );
        },

        isResolved: function() {
            return this.get("resolveProperties").resolved;
        },

        isDangling: function() {
            return this.highlight === undefined;
        },

        isDeleted: function() {
            return this.get('deleted');
        },

        // This method sets two properties on the comment model:
        // highlights: all dom elements with corresponding marker ref
        // highlight: first dom element with corresponding marker ref
        _setHighlights: function(markerRef) {
            var $highlights;
            if (markerRef !== undefined) {
                $highlights = $('#content .wiki-content:first').find('.inline-comment-marker[data-ref="' + markerRef + '"]');
            } else {
                $highlights = $('.ic-current-selection');
            }
            if ($highlights.length !== 0) {
                this.highlights = $highlights;
                this.highlight = $highlights.first();
            } else {
                this.highlights = undefined;
                this.highlight = undefined;
            }
        },

        // Override destroy function to keep the comment model is still alive in collection
        // after destroy call
        destroy: function(options) {
            options = options ? _.clone(options) : {};
            var model = this;
            var success = options.success;

            var wrapError = function(model, options) {
                var error = options.error;
                options.error = function(resp) {
                    if (error) error(model, resp, options);
                    model.trigger('error', model, resp, options);
                };
            };

            options.success = function(resp) {
                if (success) success(model, resp, options);
            };

            wrapError(this, options);

            return this.sync('delete', this, options);
        }
    });

    return Comment;
});