define('confluence/hipchat/space-to-room/internal/service', [
    'underscore',
    'jquery'
],
/**
 * Internal services used to manage space to room configuration
 *
 * N.B. The exports tag allows us to define the AMD module name that gets exported in this file.
 * @exports confluence/hipchat/space-to-room/internal/service
 */
function (
    //AJS, Don't inject AJS because in Conf 5.5.6 it is minimised. See HC-9447.
    _,
    $
) {

    'use strict';

    var Routes = {
        allRooms: function(space) {
            return AJS.Confluence.getContextPath() + '/rest/hipchat/integration/1.0/rooms?key=' + (space ? space.key : "");
        },
        roomConfig: function(spaceKey, roomId, notifName, initialLink) {
            var url = AJS.Confluence.getContextPath() + '/rest/hipchat/spacetoroom/1.0/config/' + encodeURI(spaceKey) + '/' + encodeURI(roomId);
            if(notifName) {
                url += '/' + encodeURI(notifName);
            }
            if(initialLink) {
                url += '?initialLink=true';
            }
            AJS.debug('url=', url);
            return url;
        },
        roomConfigNotifyClient: function(spaceKey, roomId) {
            var url = AJS.Confluence.getContextPath() + '/rest/hipchat/spacetoroom/1.0/configsettings/notifyclient/' + encodeURI(spaceKey) + '/' + encodeURI(roomId);
            AJS.debug('url=', url);
            return url;
        },
        createRoom: function(newRoomName) {
            return AJS.contextPath() + '/rest/hipchat/integration/1.0/rooms?roomName=' + encodeURI(newRoomName);
        }
    };

    var configService = (function() {
        return {
            removeRoomMapping: function (spaceKey, roomId) {
                return $.ajax(Routes.roomConfig(spaceKey, roomId), {
                    type: 'DELETE'
                });
            },

            toggleNotification: function (spaceKey, roomId, notifName, enable, initialLink) {
                var requestType = enable ? 'PUT' : 'DELETE';
                return $.ajax(Routes.roomConfig(spaceKey, roomId, notifName, initialLink), {
                    type: requestType
                });
            },

            toggleNotifyClient: function(spaceKey, roomId, enable) {
                var requestType = enable ? 'PUT' : 'DELETE';
                return $.ajax(Routes.roomConfigNotifyClient(spaceKey, roomId), {
                    type: requestType
                });
            }
        };
    })();

    // Encapsulate a room service and data.
    // No protecting for mutation of passed in/out object by client as I don't want to where to cost of cloning both the array
    // and objects to prevent this, after all this is an internal service anyway.
    var roomServicePromise = (function(space) {
        var allowMultipleMappingsToSameRoom = space == null;
        var roomIdsToRooms = {}; // { "roomId": { roomId: "roomId", roomName: "roomName", roomPrivate: boolean }, ... }
        var selectableRooms = []; // cache of [ { id: "roomId", roomName: "roomName", roomPrivate: boolean } ] for use by select2 query. Derived from roomIdsToRooms
        var listeners = {};

        function rebuildRoomData() {
            selectableRooms = [];
            for (var roomId in roomIdsToRooms) {
                var room = roomIdsToRooms[roomId];
                if (room.selectable) {
                    selectableRooms.push({id: room.roomId, roomName: room.roomName, isPrivate: room.roomPrivate });
                }
            }
            selectableRooms = _.sortBy(selectableRooms, function (item) { return item.roomName.toLocaleLowerCase(); });
            _.each(listeners, function(fn) {
                fn(selectableRooms);
            });
        }

        function addRoom(room, isNew) {
            if (isNew) {
                roomIdsToRooms[room.roomId] = room;
            }
            rebuildRoomData();
        }

        function handleMappingToRoomRemoved(room) {
            if (!allowMultipleMappingsToSameRoom) {
                room.selectable = true;
                var roomLocal = roomIdsToRooms[room.roomId];
                if (roomLocal) {
                    roomLocal.selectable = true;
                }
                rebuildRoomData();
            }
        }

        function handleMappingToRoomAdded(room) {
            if (!allowMultipleMappingsToSameRoom) {
                room.selectable = false;
                var roomLocal = roomIdsToRooms[room.roomId];
                if (roomLocal) {
                    roomLocal.selectable = false;
                }
                rebuildRoomData();
            }
        }

        function onChange(fn) {
            listeners[fn] = fn;
        }

        function offChange(fn) {
            delete listeners[fn];
        }

        function getRoomById(roomId) {
            return roomIdsToRooms[roomId];
        }

        function findRoomByNameIgnoreCase(roomName) {
            var roomNameLower = roomName.toLocaleLowerCase();
            for (var roomId in roomIdsToRooms) {
                var room = roomIdsToRooms[roomId];
                if (room.roomName.toLocaleLowerCase() === roomNameLower) {
                    return room;
                }
            }
            return null;
        }

        function allowSelectionByName(roomName) {
            if (allowMultipleMappingsToSameRoom) {
                return true;
            } else {
                var room = findRoomByNameIgnoreCase(roomName);
                return room != null && room.selectable;
            }
        }

        function getOrCreateRoom(room) {
            var deferred = $.Deferred();
            var existingRoom = findRoomByNameIgnoreCase(room.roomName);
            if (existingRoom) {
                deferred.resolve(existingRoom);
                return deferred.promise();
            } else {
                $.ajax({
                    url: Routes.createRoom(room.roomName),
                    type: 'POST',
                    dataType: 'json',
                    contentType: 'application/json',
                    cache: false
                }).done(function (hipChatRoomDefinition) {
                    var createdRoom = {
                        roomId: hipChatRoomDefinition.id,
                        roomName: hipChatRoomDefinition.name,
                        roomPrivate: hipChatRoomDefinition["private"], // Note: YUI compressor complains because private is a keyword.
                        selectable: true
                    };
                    roomIdsToRooms[createdRoom.roomId] = createdRoom;
                    rebuildRoomData();
                    deferred.resolve(createdRoom);
                }).fail(function (xhr) {
                    deferred.reject(room.roomName, xhr.status, xhr.statusText)
                })
            }
            return deferred.promise();
        }

        function getSelectableRooms() {
            return selectableRooms;
        }

        var roomService = {
            handleMappingToRoomRemoved: handleMappingToRoomRemoved,
            handleMappingToRoomAdded: handleMappingToRoomAdded,
            getRoomById: getRoomById,
            findRoomByNameIgnoreCase: findRoomByNameIgnoreCase,
            allowSelectionByName: allowSelectionByName,
            getOrCreateRoom: getOrCreateRoom,
            getSelectableRooms: getSelectableRooms,
            onChange: onChange,
            offChange: offChange
        };

        var roomsAvailablePromise = (function () {
            var roomLoader = $.Deferred();
            // Note that we set cache to false to prevent IE9 caching previous results. See CONFDEV-29008.
            $.ajax({
                url: Routes.allRooms(space),
                dataType: 'json',
                cache: false
            }).done(function(data) {
                roomIdsToRooms = _.object(_.map(data, function (item) {
                    return [item.id, {
                        roomId: item.id,
                        roomName: item.name,
                        roomPrivate: item["private"],
                        selectable: true
                    }];
                }));

                if (space) {
                    // remove rooms already mapped
                    $('.hipchat-room-config').each(function (index, el) {
                        var roomId = $(el).attr('data-room-id');
                        if (roomId) {
                            var room = roomIdsToRooms[roomId];
                            if (room) {
                                room.selectable = false;
                                rebuildRoomData();
                            }
                        }
                    });
                }

                rebuildRoomData();
                roomLoader.resolve(roomService);
            }).fail(function (xhr) {
                roomLoader.reject(xhr);
            });
            return roomLoader.promise();
        })();

        return roomsAvailablePromise;
    });

    return function (space) {
        return {
            configService: configService,
            roomServicePromise: roomServicePromise(space)
        }
    };
});
