package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.plugin.descriptor.web.DefaultWebInterfaceContext;
import com.atlassian.confluence.plugin.descriptor.web.WebInterfaceContext;
import com.atlassian.confluence.plugins.createcontent.BlueprintConstants;
import com.atlassian.confluence.plugins.createcontent.BlueprintStateController;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.SpaceBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.SpaceBlueprintStateController;
import com.atlassian.confluence.plugins.createcontent.extensions.BlueprintModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.extensions.UserBlueprintConfigManager;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.plugins.createcontent.impl.PluginBackedBlueprint;
import com.atlassian.confluence.plugins.createcontent.impl.SpaceBlueprint;
import com.atlassian.confluence.plugins.createcontent.model.BlueprintState;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateDialogWebItemEntity;
import com.atlassian.confluence.plugins.createcontent.services.BlueprintDiscoveryService;
import com.atlassian.confluence.plugins.createcontent.services.BlueprintSorter;
import com.atlassian.confluence.plugins.dialog.wizard.api.DialogManager;
import com.atlassian.confluence.plugins.dialog.wizard.api.DialogWizard;
import com.atlassian.confluence.plugins.dialog.wizard.api.DialogWizardEntity;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.util.i18n.DocumentationBean;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.web.WebInterfaceManager;
import com.atlassian.plugin.web.descriptors.WebItemModuleDescriptor;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.BLANK_PAGE_BLUEPRINT;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.BLOG_POST_BLUEPRINT;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.CREATE_DIALOG_CONTENT_SECTION;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.CREATE_SPACE_DIALOG_CONTENT_SECTION;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.BLUEPRINT_MODULE_KEY;
import static com.google.common.collect.Lists.newArrayList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.slf4j.LoggerFactory.getLogger;

public class DefaultBlueprintWebItemService implements BlueprintWebItemService {
    private static final Logger log = getLogger(DefaultBlueprintWebItemService.class);

    private final BlueprintStateController contentBlueprintStateController;
    private final SpaceBlueprintStateController spaceBlueprintStateController;
    private final WebInterfaceManager webInterfaceManager;
    private final PermissionManager permissionManager;
    private final UserBlueprintConfigManager userBlueprintConfigManager;
    private final ContentBlueprintManager contentBlueprintManager;
    private final IconUrlProvider iconUrlProvider;
    private final PluginAccessor pluginAccessor;
    private final PageTemplateWebItemService pageTemplateWebItemService;
    private final SpaceBlueprintManager spaceBlueprintManager;
    private final BlueprintDiscoveryService blueprintDiscoverer;
    private final BlueprintSorter blueprintSorter;
    private final DialogManager dialogManager;

    public DefaultBlueprintWebItemService(final BlueprintStateController contentBlueprintStateController,
                                          final SpaceBlueprintStateController spaceBlueprintStateController,
                                          final WebInterfaceManager webInterfaceManager,
                                          final PermissionManager permissionManager,
                                          final UserBlueprintConfigManager userBlueprintConfigManager,
                                          final ContentBlueprintManager contentBlueprintManager,
                                          IconUrlProvider iconUrlProvider, final PluginAccessor pluginAccessor,
                                          PageTemplateWebItemService pageTemplateWebItemService,
                                          @Qualifier("spaceBlueprintManager") final SpaceBlueprintManager spaceBlueprintManager,
                                          BlueprintDiscoveryService blueprintDiscoverer, BlueprintSorter blueprintSorter, DialogManager dialogManager) {
        this.contentBlueprintStateController = contentBlueprintStateController;
        this.spaceBlueprintStateController = spaceBlueprintStateController;
        this.webInterfaceManager = webInterfaceManager;
        this.permissionManager = permissionManager;
        this.userBlueprintConfigManager = userBlueprintConfigManager;
        this.contentBlueprintManager = contentBlueprintManager;
        this.iconUrlProvider = iconUrlProvider;
        this.pluginAccessor = pluginAccessor;
        this.pageTemplateWebItemService = pageTemplateWebItemService;
        this.spaceBlueprintManager = spaceBlueprintManager;
        this.blueprintDiscoverer = blueprintDiscoverer;
        this.blueprintSorter = blueprintSorter;
        this.dialogManager = dialogManager;
    }

    @Override
    public List<CreateDialogWebItemEntity> getCreateContentWebItems(final Space space, final I18NBean i18NBean, final DocumentationBean documentationBean, final ConfluenceUser user) {
        // Sort alphabetically with Blank Page and Blog Post at top. By web-item weight the Blank Page and Blog items
        // should be the first two.
        List<CreateDialogWebItemEntity> pluginItems = getPluginItems(i18NBean, documentationBean, space, user);

        pluginItems.addAll(pageTemplateWebItemService.getPageTemplateItems(space, user));

        return filterAndSortItems(space, user, pluginItems);
    }

    private List<CreateDialogWebItemEntity> filterAndSortItems(@Nonnull Space space, @Nullable ConfluenceUser user,
                                                               @Nonnull List<CreateDialogWebItemEntity> pluginItems) {
        // TODO Use conditions on the webitems instead of checking permissions here
        // Users might only be able to create Pages, or only BlogPosts. They will be able to create at least one type,
        // or they would not get the Create button at all.
        final boolean canCreatePages = canCreatePages(space, user);
        final boolean canCreateBlogPosts = canCreateBlogPosts(space, user);
        if (!canCreatePages && !canCreateBlogPosts)
            return newArrayList();  // hugely unlikely but makes Alice happy

        final CreateDialogWebItemEntity blankPageItem = pluginItems.get(0);
        final CreateDialogWebItemEntity blogPostItem = pluginItems.get(1);
        // FIXME: This is UGLY!
        blankPageItem.setContentBlueprintId(BLANK_PAGE_BLUEPRINT.getId());
        blogPostItem.setContentBlueprintId(BLOG_POST_BLUEPRINT.getId());

        if (!canCreatePages)
            return Lists.newArrayList(blogPostItem);

        if (!canCreateBlogPosts)
            pluginItems.remove(blogPostItem);

        // The assumption is that all templates and Blueprints will add pages...
        return blueprintSorter.sortContentBlueprintItems(pluginItems, space, user);
    }

    @Override
    public List<CreateDialogWebItemEntity> getCreateSpaceWebItems(final I18NBean i18NBean, final DocumentationBean documentationBean, final ConfluenceUser user) {
        List<CreateDialogWebItemEntity> pluginItems = getPluginItems(i18NBean, documentationBean, null, user);

        return blueprintSorter.sortSpaceBlueprintItems(pluginItems, user);
    }

    /**
     * Retrieves the blueprint module descriptor given a web-item module descriptor that specifies the {@link com.atlassian.confluence.plugins.createcontent.BlueprintConstants#BLUEPRINT_PARAM_KEY} as a param.
     *
     * @param webItemDescriptor the web-item module descriptor that should have a param with key {@link com.atlassian.confluence.plugins.createcontent.BlueprintConstants#BLUEPRINT_PARAM_KEY}
     * @return the blueprint module descriptor
     */
    @Override
    @Deprecated
    public BlueprintModuleDescriptor getBlueprintDescriptorForWebItem(final ModuleDescriptor webItemDescriptor) {
        final String blueprintKey = getBlueprintModuleKey(webItemDescriptor);
        if (isBlank(blueprintKey))
            return null;

        final ModuleCompleteKey blueprintCompleteModuleKey = new ModuleCompleteKey(webItemDescriptor.getPluginKey(), blueprintKey);
        final ModuleDescriptor blueprintModuleDescriptor = pluginAccessor.getEnabledPluginModule(blueprintCompleteModuleKey.getCompleteKey());

        if (!(blueprintModuleDescriptor instanceof BlueprintModuleDescriptor)) {
            log.debug("{} is not a <blueprint>.", blueprintCompleteModuleKey);
            return null;
        }
        return (BlueprintModuleDescriptor) blueprintModuleDescriptor;
    }

    // HACK! FIXME: This is a hack to get only the personal space web item. We need a way to use conditions to do this for us when calling webInterfaceManager.getDisplayableItems()
    @Override
    public List<CreateDialogWebItemEntity> getCreatePersonalSpaceWebItems(final I18NBean i18NBean, final DocumentationBean documentationBean, final ConfluenceUser remoteUser) {
        WebItemModuleDescriptor webItem = (WebItemModuleDescriptor) pluginAccessor.getPluginModule(BlueprintConstants.MODULE_KEY_PERSONAL_SPACE_ITEM);
        SpaceBlueprint pluginBackedBlueprint = getSpaceBlueprintForWebItem(webItem);

        final String name = getDisplayableWebItemName(i18NBean, webItem);

        final List<CreateDialogWebItemEntity> webItemEntities = newArrayList();
        webItemEntities.add(createWebItemEntity(i18NBean, webItem, name, Collections.<UUID>emptySet(), false, pluginBackedBlueprint));
        return webItemEntities;
    }

    private boolean canCreateBlogPosts(final Space space, final ConfluenceUser remoteUser) {
        return permissionManager.hasCreatePermission(remoteUser, space, BlogPost.class);
    }

    private boolean canCreatePages(final Space space, final ConfluenceUser remoteUser) {
        return permissionManager.hasCreatePermission(remoteUser, space, Page.class);
    }

    /**
     * @param context the context in which the create dialog web items will be displayed in
     * @return the list of web item module descriptors for the create dialog
     */
    @Override
    public List<WebItemModuleDescriptor> getCreateDialogWebItemModuleDescriptors(WebInterfaceContext context) {
        return webInterfaceManager.getDisplayableItems(CREATE_DIALOG_CONTENT_SECTION, context.toMap());
    }

    /**
     * @return A mutable list of create dialog web item entities that should be shown to the user.
     */
    private List<CreateDialogWebItemEntity> getPluginItems(final I18NBean i18NBean, final DocumentationBean documentationBean,
                                                           final Space space, ConfluenceUser user) {
        final boolean isPageBp = space != null;
        final String section = isPageBp ? CREATE_DIALOG_CONTENT_SECTION : CREATE_SPACE_DIALOG_CONTENT_SECTION;
        final Map<UUID, BlueprintState> blueprintStateMap = isPageBp ?
                contentBlueprintStateController.getAllContentBlueprintState(section, user, space) :
                spaceBlueprintStateController.getAllSpaceBlueprintState(section, user);

        // Anonymous users cannot skip the how-to page (sorry).
        final Set<UUID> skipHowToUseKeys = user != null ? userBlueprintConfigManager.getSkipHowToUseKeys(user) : Collections.<UUID>emptySet();

        // Gets all enabled displayable web-items for create content section
        final DefaultWebInterfaceContext webInterfaceContext = new DefaultWebInterfaceContext();
        webInterfaceContext.setSpace(space);
        webInterfaceContext.setCurrentUser(user);
        List<WebItemModuleDescriptor> displayableItems = webInterfaceManager.getDisplayableItems(section, webInterfaceContext.toMap());

        final List<CreateDialogWebItemEntity> webItems = newArrayList();
        for (final WebItemModuleDescriptor webItem : displayableItems) {
            // Translate i18n keys as the Soy getText function only works with string literals
            // (i.e. 'text' and not $i18nKey even though $i18nKey is a string)
            final String name = getDisplayableWebItemName(i18NBean, webItem);
            if (isBlank(name)) {
                log.warn(webItem + " not added as it is missing a displayable name.");
                continue;
            }

            String dialogWizardKey = webItem.getParams().get("dialogWizardKey");
            if (StringUtils.isNotBlank(dialogWizardKey)) {
                DialogWizard dialogWizard = dialogManager.getDialogWizardByKey(dialogWizardKey);
                if (dialogWizard != null) {
                    webItems.add(createWebItemEntity(name, i18NBean, webItem, dialogWizard));
                    continue;
                }
            }

            // Skip this blueprint if we have a 'blueprintKey' parameter
            PluginBackedBlueprint blueprint = isPageBp ?
                    getContentBlueprintForWebItem(webItem, space) :
                    getSpaceBlueprintForWebItem(webItem);

            // Some web items may not be blueprints, so they won't be in the state map. We choose to display them.
            if (blueprint != null) {
                BlueprintState blueprintState = blueprintStateMap.get(blueprint.getId());
                if (blueprintState != null && !BlueprintState.FULLY_ENABLED.equals(blueprintState))
                    continue;
            }

            webItems.add(createWebItemEntity(i18NBean, webItem, name, skipHowToUseKeys, isPageBp, blueprint));
        }

        // Anonymous users will NOT get the Blueprints discover experience
        if (AuthenticatedUserThreadLocal.get() != null)
            return blueprintDiscoverer.discoverRecentlyInstalled(webItems);

        return webItems;
    }

    // create a non blueprint create dialog web item
    private CreateDialogWebItemEntity createWebItemEntity(final String name,
                                                          final I18NBean i18NBean,
                                                          final WebItemModuleDescriptor webItem,
                                                          final DialogWizard dialogWizard) {

        final String description = i18NBean.getText(webItem.getDescriptionKey());
        final String iconURL = iconUrlProvider.getIconURL(webItem);

        DialogWizardEntity dialogWizardEntity = new DialogWizardEntity(i18NBean, dialogWizard);
        return new CreateDialogWebItemEntity(name, description, webItem.getStyleClass(), iconURL,
                webItem.getCompleteKey(), null, null, null, false,
                dialogWizardEntity);
    }

    // create a space/page blueprint web item
    private CreateDialogWebItemEntity createWebItemEntity(final I18NBean i18NBean,
                                                          final WebItemModuleDescriptor webItem, final String entityName,
                                                          Set<UUID> skipHowToUseMap, final boolean contentBp, PluginBackedBlueprint pluginBackedBlueprint) {
        final String description = i18NBean.getText(webItem.getDescriptionKey());
        final String iconURL = iconUrlProvider.getIconURL(webItem);

        // WebItems may or may not have a creator - if no creator-key is specified in the web-item we assume
        // that the item registers custom JS code to perform the content creation.
        UUID contentBlueprintUuid = null;
        String createResult = null;
        String howToUseTemplate = null;
        boolean skipHowToUse = false;
        DialogWizardEntity dialogWizardEntity = null;
        String moduleCompleteKey = null;
        String directLink = webItem.getParams().get("directLink");

        DialogWizard dialogWizard = null;
        if (contentBp) {
            final ContentBlueprint contentBlueprint = (ContentBlueprint) pluginBackedBlueprint;
            if (contentBlueprint != null) {
                contentBlueprintUuid = contentBlueprint.getId();
                createResult = contentBlueprint.getCreateResult();
                howToUseTemplate = contentBlueprint.getHowToUseTemplate();
                skipHowToUse = skipHowToUseMap.contains(contentBlueprint.getId());
                dialogWizard = contentBlueprint.getDialogWizard();
                moduleCompleteKey = contentBlueprint.getModuleCompleteKey();
            }
        } else {
            // retrieving Space Bps
            final SpaceBlueprint spaceBlueprint = (SpaceBlueprint) pluginBackedBlueprint;
            if (spaceBlueprint != null) {
                contentBlueprintUuid = spaceBlueprint.getId();
                // TODO - maybe add this later :P
//                howToUseTemplate = spaceBlueprint.getHowToUseTemplate();
                createResult = "space";
                skipHowToUse = skipHowToUseMap.contains(spaceBlueprint.getId());

                dialogWizard = spaceBlueprint.getDialogWizard();
                moduleCompleteKey = spaceBlueprint.getModuleCompleteKey();
            }
        }
        if (dialogWizard != null) {
            dialogWizardEntity = new DialogWizardEntity(i18NBean, dialogWizard);
        }

        CreateDialogWebItemEntity entity = new CreateDialogWebItemEntity(entityName, description, webItem.getStyleClass(), iconURL,
                webItem.getCompleteKey(), contentBlueprintUuid, createResult, howToUseTemplate, directLink, skipHowToUse,
                dialogWizardEntity);
        entity.setBlueprintModuleCompleteKey(moduleCompleteKey);
        return entity;
    }

    private static String getDisplayableWebItemName(final I18NBean i18NBean, final WebItemModuleDescriptor webItem) {
        if (isNotBlank(webItem.getI18nNameKey())) {
            return i18NBean.getText(webItem.getI18nNameKey());
        } else if (isNotBlank(webItem.getName())) {
            return webItem.getName();
        }
        return null;
    }

    @Nullable
    private ContentBlueprint getContentBlueprintForWebItem(ModuleDescriptor webItemModuleDescriptor, Space space) {
        final String blueprintKey = getBlueprintModuleKey(webItemModuleDescriptor);
        if (isBlank(blueprintKey))
            return null;

        final ModuleCompleteKey completeKey = new ModuleCompleteKey(webItemModuleDescriptor.getPluginKey(), blueprintKey);
        return contentBlueprintManager.getPluginBackedContentBlueprint(completeKey, space.getKey());
    }

    @Nullable
    private SpaceBlueprint getSpaceBlueprintForWebItem(ModuleDescriptor webItemModuleDescriptor) {
        final String blueprintKey = getBlueprintModuleKey(webItemModuleDescriptor);
        if (isBlank(blueprintKey))
            return null;

        final ModuleCompleteKey completeKey = new ModuleCompleteKey(webItemModuleDescriptor.getPluginKey(), blueprintKey);
        return spaceBlueprintManager.getCloneByModuleCompleteKey(completeKey);
    }

    private String getBlueprintModuleKey(ModuleDescriptor webItemModuleDescriptor) {
        final Map<String, String> moduleDescriptorParams = webItemModuleDescriptor.getParams();
        final String blueprintKey = moduleDescriptorParams.get(BLUEPRINT_MODULE_KEY.key());

        if (isBlank(blueprintKey)) {
            log.debug("No param with key '{}' specified in module descriptor with key '{}'", BLUEPRINT_MODULE_KEY.key(), webItemModuleDescriptor.getCompleteKey());
        }
        return blueprintKey;
    }
}
