package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.PluginSpaceBlueprintAccessor;
import com.atlassian.confluence.plugins.createcontent.activeobjects.ContentTemplateRefAo;
import com.atlassian.confluence.plugins.createcontent.extensions.ContentTemplateModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.extensions.SpaceBlueprintModuleDescriptor;
import com.atlassian.confluence.plugins.dialog.wizard.api.DialogWizard;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.sal.api.message.I18nResolver;
import com.google.common.collect.Lists;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;
import java.util.UUID;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Returns SpaceBlueprints from the plugin system.
 *
 * @since 5.0
 */
public class PluginSpaceBlueprintManager implements PluginSpaceBlueprintAccessor {
    private final PluginAccessor pluginAccessor;
    private final ContentTemplateRefManager contentTemplateRefManager;
    private final I18nResolver i18nResolver;

    private final I18NBeanFactory i18NBeanFactory;
    private final LocaleManager localeManager;

    public PluginSpaceBlueprintManager(PluginAccessor pluginAccessor,
                                       ContentTemplateRefManager contentTemplateRefManager,
                                       I18nResolver i18nResolver, I18NBeanFactory i18NBeanFactory, LocaleManager localeManager) {
        this.pluginAccessor = pluginAccessor;
        this.contentTemplateRefManager = contentTemplateRefManager;
        this.i18nResolver = i18nResolver;
        this.i18NBeanFactory = i18NBeanFactory;
        this.localeManager = localeManager;
    }

    @Nonnull
    @Override
    public SpaceBlueprint getByModuleCompleteKey(@Nonnull ModuleCompleteKey moduleCompleteKey) {
        // NOTE - the SpaceBPs coming from the plugin system are not the same as what we want to pass around.
        // Plugin BPs can only know about plugin-key references, but persisted BPs can know about user-created
        // templates ("simple Blueprints"). We need to convertToModuleId from one to the other. dT
        SpaceBlueprintModuleDescriptor descriptor = getDescriptor(moduleCompleteKey);
        checkNotNull(descriptor, "Space Blueprint module descriptor not found [key='" + moduleCompleteKey + "']");

        return getSpaceBlueprint(descriptor);
    }

    @Nullable
    @Override
    public DialogWizard getDialogByModuleCompleteKey(@Nonnull final ModuleCompleteKey moduleCompleteKey) {
        // TODO - dialog wizard - should come directly from the respective Plugin Blueprint if there is one
        final SpaceBlueprintModuleDescriptor desc = getDescriptor(moduleCompleteKey);
        if (desc != null) {
            return desc.getDialogWizard();
        }
        return null;
    }

    @Nonnull
    @Override
    public List<SpaceBlueprint> getAll() {
        List<SpaceBlueprintModuleDescriptor> descriptors = pluginAccessor.getEnabledModuleDescriptorsByClass(
                SpaceBlueprintModuleDescriptor.class);

        List<SpaceBlueprint> blueprints = Lists.newArrayList();
        for (SpaceBlueprintModuleDescriptor descriptor : descriptors) {
            blueprints.add(getSpaceBlueprint(descriptor));
        }

        return blueprints;
    }

    @Nullable
    private SpaceBlueprintModuleDescriptor getDescriptor(@Nonnull final ModuleCompleteKey moduleCompleteKey) {
        final String key = moduleCompleteKey.getCompleteKey();
        return (SpaceBlueprintModuleDescriptor) pluginAccessor.getEnabledPluginModule(key);
    }

    private SpaceBlueprint getSpaceBlueprint(SpaceBlueprintModuleDescriptor moduleDescriptor) {
        UUID homePageId = null;
        final SpaceBlueprintModuleDescriptor.ContentTemplateRefNode homePageRefNode =
                moduleDescriptor.getContentTemplateRefNode();
        if (homePageRefNode != null) {
            // If no custom space home is specified, the Default homepage specified in Site Admin will be left alone
            // when the space is created.
            homePageId = storeContentTemplateRef(homePageRefNode);
        }
        DialogWizard dialogWizard = moduleDescriptor.getDialogWizard();
        checkNotNull(dialogWizard,
                "Space Blueprint dialog-wizard not defined [key='" + moduleDescriptor.getCompleteKey() + "']");

        String i18nNameKey = moduleDescriptor.getI18nNameKey();
        checkNotNull(i18nNameKey,
                "Space Blueprint i18n-name-key not defined [key='" + moduleDescriptor.getCompleteKey() + "']");

        List<ModuleCompleteKey> promotedBlueprintKeys = moduleDescriptor.getPromotedBlueprintKeys();

        final SpaceBlueprint spaceBlueprint = new SpaceBlueprint(null, moduleDescriptor.getCompleteKey(),
                i18nResolver.getText(i18nNameKey), true, promotedBlueprintKeys, dialogWizard,
                moduleDescriptor.getCategory());
        spaceBlueprint.setHomePageId(homePageId);
        return spaceBlueprint;
    }

    /**
     * This will CREATE a new template ref every time.
     */
    @Nonnull
    private UUID storeContentTemplateRef(@Nonnull SpaceBlueprintModuleDescriptor.ContentTemplateRefNode refNode) {
        ContentTemplateRefAo ao = storeContentTemplateRefAo(refNode);
        return UUID.fromString(ao.getUuid());
    }

    @Nonnull
    private ContentTemplateRefAo storeContentTemplateRefAo(
            @Nonnull SpaceBlueprintModuleDescriptor.ContentTemplateRefNode refNode) {
        String moduleCompleteKey = refNode.ref.getCompleteKey();
        ContentTemplateModuleDescriptor contentTemplateDescriptor =
                (ContentTemplateModuleDescriptor) pluginAccessor.getEnabledPluginModule(moduleCompleteKey);

        String i18nNameKey = contentTemplateDescriptor.getI18nNameKey();
        I18NBean i18nBean = i18NBeanFactory.getI18NBean(localeManager.getSiteDefaultLocale());
        String name = i18nBean.getText(i18nNameKey);
        final ContentTemplateRef contentTemplateRef = new ContentTemplateRef(null, 0L, moduleCompleteKey, name, true, null);

        final ContentTemplateRefAo contentTemplateRefAo = contentTemplateRefManager.createAo(contentTemplateRef);

        if (refNode.children != null) {
            for (SpaceBlueprintModuleDescriptor.ContentTemplateRefNode child : refNode.children) {
                final ContentTemplateRefAo childAo = storeContentTemplateRefAo(child);
                childAo.setParent(contentTemplateRefAo);
                childAo.save();
            }
        }

        return contentTemplateRefAo;
    }
}
