package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.confluence.plugins.createcontent.activeobjects.PluginBackedBlueprintAo;
import com.atlassian.sal.api.transaction.TransactionCallback;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.UUID;

public class HelperAoManager<A extends PluginBackedBlueprintAo> {
    static interface HelperCallback<A extends PluginBackedBlueprintAo> {
        public void onDelete(A ao);

        public void onDeleteAll(A[] aos);
    }

    private final ActiveObjects activeObjects;
    private final Class<A> aoClass;

    public HelperAoManager(@Nonnull ActiveObjects activeObjects, @Nonnull Class<A> aoClass) {
        this.activeObjects = activeObjects;
        this.aoClass = aoClass;
    }

    @Nonnull
    public A createWithUuid() {
        final A newBp = activeObjects.create(aoClass);
        newBp.setUuid(UUID.randomUUID().toString());
        return newBp;
    }

    public boolean delete(@Nonnull final UUID id, @Nonnull final HelperCallback<A> callback) {
        return activeObjects.executeInTransaction(new TransactionCallback<Boolean>() {
            @Override
            public Boolean doInTransaction() {
                // AO deleteWithSQL() doesn't work for some reason ("column not found!" error)
                A ao = internalGetAoById(id);
                if (ao != null) {
                    callback.onDelete(ao);
                    activeObjects.delete(ao);
                }

                return ao != null;
            }
        });
    }

    public void delete(@Nonnull final A object, @Nonnull final HelperCallback<A> callback) {
        activeObjects.executeInTransaction(new TransactionCallback<Void>() {
            @Override
            public Void doInTransaction() {
                callback.onDelete(object);
                activeObjects.delete(object);
                return null;
            }
        });
    }

    public int deleteAll(@Nonnull final HelperCallback<A> callback) {
        return activeObjects.executeInTransaction(new TransactionCallback<Integer>() {
            @Override
            public Integer doInTransaction() {
                A[] results = activeObjects.find(aoClass);
                callback.onDeleteAll(results);
                activeObjects.delete(results);
                return results.length;
            }
        });
    }

    @Nullable
    public A internalGetAoById(@Nonnull UUID id) {
        A[] aos = activeObjects.find(aoClass, "UUID = ?", id.toString());
        if (aos != null && aos.length > 1) {
            throw new RuntimeException("More than one object with the same UUID! (" + id.toString() + ")");
        }
        return aos != null && aos.length > 0 ? aos[0] : null;
    }
}
