package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.core.ContentPropertyManager;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateBlueprintPageRestEntity;
import com.atlassian.confluence.plugins.createcontent.services.RequestStorage;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageEntity;
import com.atlassian.sal.api.web.context.HttpContext;
import org.codehaus.jackson.map.ObjectMapper;

import javax.servlet.http.HttpSession;
import java.io.IOException;

import static org.apache.commons.lang3.StringUtils.isBlank;

/*
    This storage originally used a contentPropertyManager and stored against the Draft id, but this doesn't
    work for Drafts by anonymous users (the same draft with id 0 would be used for all anonymous storage).
    Given that all create-request drafts are expected to be converted to pages within the same session we can
    use the Session for storage.

    If we decide to reuse the contentPropertyManager later (CONFDEV-18341), be sure to look at old versions of
    this class!
 */
public class DefaultRequestStorage implements RequestStorage {
    private static final String DRAFT_CREATE_REQUEST = "create.blueprint.page.draft.request";

    private final HttpContext httpContext;
    private final ContentPropertyManager contentPropertyManager;
    private final ObjectMapper objectMapper;

    public DefaultRequestStorage(HttpContext httpContext, ContentPropertyManager contentPropertyManager) {
        this.httpContext = httpContext;
        this.contentPropertyManager = contentPropertyManager;
        objectMapper = new ObjectMapper();
    }

    @Override
    public void storeCreateRequest(CreateBlueprintPageEntity createRequest, ContentEntityObject ceo) {
        if (ceo.isPersistent()) {
            String jsonRequest;
            try {
                jsonRequest = objectMapper.writeValueAsString(createRequest);
                contentPropertyManager.setTextProperty(ceo, DRAFT_CREATE_REQUEST, jsonRequest);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        } else {
            getSession().setAttribute(getAttributeKey(ceo), createRequest);
        }
    }

    @Override
    public CreateBlueprintPageEntity retrieveRequest(ContentEntityObject ceo) {
        CreateBlueprintPageEntity request;

        if (ceo.isPersistent()) {
            String jsonRequest = contentPropertyManager.getTextProperty(ceo, DRAFT_CREATE_REQUEST);
            if (isBlank(jsonRequest))
                throw new IllegalStateException("No persisted CreateBlueprint request found for draft with id: " + ceo.getId());

            try {
                request = objectMapper.readValue(jsonRequest, CreateBlueprintPageRestEntity.class);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        } else {
            request = (CreateBlueprintPageEntity) getSession().getAttribute(getAttributeKey(ceo));
        }

        return request;
    }

    // TODO - This method can be deleted once reliable-save has been finalised
    // See https://jira.atlassian.com/browse/CONFDEV/component/37892
    @Override
    public void clear(ContentEntityObject ceo) {
        if (ceo.isPersistent()) {
            contentPropertyManager.removeProperty(ceo, DRAFT_CREATE_REQUEST);
        } else {
            getSession().removeAttribute(getAttributeKey(ceo));
        }
    }

    private String getAttributeKey(ContentEntityObject ceo) {
        return DRAFT_CREATE_REQUEST + ceo.getIdAsString();
    }

    private HttpSession getSession() {
        return httpContext.getSession(true);
    }
}