package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.bandana.BandanaManager;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.SpaceBandanaContext;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.services.BlueprintResolver;
import com.atlassian.confluence.plugins.createcontent.services.PromotedBlueprintService;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.BLANK_PAGE_BLUEPRINT;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.BLOG_POST_BLUEPRINT;
import static com.atlassian.confluence.plugins.createcontent.rest.BlueprintStateResource.NON_SPACE_ADMIN_PERMISSION_DENIED_RESPONSE;

public class DefaultPromotedBlueprintService implements PromotedBlueprintService {
    private final ContentBlueprintManager contentBlueprintManager;
    private final BandanaManager bandanaManager;
    private final SpaceManager spaceManager;
    private final PermissionManager permissionManager;
    private final BlueprintResolver resolver;

    public static final String KEY_PROMOTED_BPS = "promotedBps";
    private static final org.slf4j.Logger log = LoggerFactory.getLogger(DefaultPromotedBlueprintService.class);

    public DefaultPromotedBlueprintService(ContentBlueprintManager contentBlueprintManager,
                                           BandanaManager bandanaManager, SpaceManager spaceManager,
                                           PermissionManager permissionManager, BlueprintResolver resolver) {
        this.contentBlueprintManager = contentBlueprintManager;
        this.bandanaManager = bandanaManager;
        this.spaceManager = spaceManager;
        this.permissionManager = permissionManager;
        this.resolver = resolver;
    }

    @Override
    @Nonnull
    public Collection<ContentBlueprint> getPromotedBlueprints(@Nonnull Collection<ContentBlueprint> contentBlueprints,
                                                              @Nullable Space space) {
        Collection<ContentBlueprint> promotedBlueprints = Lists.newArrayList();
        Collection<ContentBlueprint> blueprintsInSpace = Lists.newArrayList(contentBlueprints);
        if (space == null)
            return promotedBlueprints;

        List<String> promotedBpsStringUuids = retrievePromotedIds(space);

        if (promotedBpsStringUuids.isEmpty())
            return promotedBlueprints;

        // special case - add blank page and blog post
        blueprintsInSpace.add(BLOG_POST_BLUEPRINT);
        blueprintsInSpace.add(BLANK_PAGE_BLUEPRINT);

        for (UUID id : convertStringToUuid(promotedBpsStringUuids)) {
            ContentBlueprint blueprint = matchIdWithBlueprint(id, blueprintsInSpace);

            if (blueprint != null)
                promotedBlueprints.add(blueprint);
        }

        return promotedBlueprints;
    }

    @Override
    public boolean promoteBlueprint(@Nonnull String blueprintId, @Nonnull String spaceKey)
            throws BlueprintIllegalArgumentException {
        return toggleBlueprintPromotion(blueprintId, spaceKey, true);
    }

    @Override
    public boolean demoteBlueprint(@Nonnull String blueprintId, @Nonnull String spaceKey)
            throws BlueprintIllegalArgumentException {
        return toggleBlueprintPromotion(blueprintId, spaceKey, false);
    }

    private boolean toggleBlueprintPromotion(@Nonnull String blueprintId, @Nonnull String spaceKey, boolean promote)
            throws BlueprintIllegalArgumentException {
        if (StringUtils.isBlank(blueprintId))
            throw new BlueprintIllegalArgumentException("Blueprint UUID is required to promote/demote blueprint", ResourceErrorType.PARAMETER_MISSING, "blueprintId");

        if (StringUtils.isBlank(spaceKey))
            throw new BlueprintIllegalArgumentException("Space key is required to promote/demote blueprint with id: " + blueprintId, ResourceErrorType.PARAMETER_MISSING, "spaceKey");

        final Space space = spaceManager.getSpace(spaceKey);

        if (space == null)
            throw new BlueprintIllegalArgumentException("Space with key '" + spaceKey + "' could not be found.", ResourceErrorType.NOT_FOUND_SPACE, spaceKey);

        if (!permissionManager.hasPermission(AuthenticatedUserThreadLocal.get(), Permission.ADMINISTER, space))
            throw new BlueprintIllegalArgumentException(NON_SPACE_ADMIN_PERMISSION_DENIED_RESPONSE, ResourceErrorType.PERMISSION_USER_ADMIN_SPACE, spaceKey);


        //check UUID is valid and points to a blueprint
        if (contentBlueprintManager.getById(UUID.fromString(blueprintId)) == null) {
            throw new BlueprintIllegalArgumentException("Valid blueprint UUID is required to promote/demote blueprint", ResourceErrorType.INVALID_ID_BLUEPRINT, blueprintId);
        }

        Set<String> promotedBpsIds = Sets.newHashSet(retrievePromotedIds(space));

        // The add/remove might not do anything if the id has already been added/removed (i.e. the method call is stale)
        boolean didSomething;
        if (promote)
            didSomething = promotedBpsIds.add(blueprintId);
        else
            didSomething = promotedBpsIds.remove(blueprintId) || removeStaleId(blueprintId, promotedBpsIds, spaceKey);

        if (didSomething)
            storePromotedIds(space, promotedBpsIds);

        return didSomething;
    }

    private boolean removeStaleId(String staleId, Set<String> promotedBpsIds, String spaceKey) {
        // The stored Blueprint Id might not be the correct one - the value in Bandana might be "stale" if the
        // Blueprint it points to has been overridden at the space or global levels since the Id was added.
        for (String promotedBpsId : promotedBpsIds) {
            ContentBlueprint newerBlueprint = resolver.resolveContentBlueprint(promotedBpsId, spaceKey);
            if (newerBlueprint.getId().toString().equals(staleId)) {
                return promotedBpsIds.remove(promotedBpsId);
            }
        }
        return false;
    }

    @Override
    public void promoteBlueprints(@Nonnull List<String> blueprintIds, @Nonnull Space space) {
        List<String> ids = retrievePromotedIds(space);
        Set<String> promotedBpsIds = Sets.newHashSet(ids);

        promotedBpsIds.addAll(blueprintIds);

        storePromotedIds(space, promotedBpsIds);
    }

    private void storePromotedIds(Space space, Set<String> promotedBpsIds) {
        SpaceBandanaContext context = new SpaceBandanaContext(space);
        bandanaManager.setValue(context, KEY_PROMOTED_BPS, Lists.newArrayList(promotedBpsIds));
    }

    @SuppressWarnings("unchecked")
    @Nonnull
    private List<String> retrievePromotedIds(Space space) {
        SpaceBandanaContext context = new SpaceBandanaContext(space);
        List<String> ids = (List<String>) bandanaManager.getValue(context, KEY_PROMOTED_BPS);
        if (ids == null)
            ids = Lists.newArrayList();

        return ids;
    }

    private ContentBlueprint matchIdWithBlueprint(@Nonnull UUID id, @Nonnull Collection<ContentBlueprint> contentBlueprints) {
        // Try match uuid to blueprints in the passed in list
        ContentBlueprint blueprint = getBlueprintFromListById(id, contentBlueprints);
        if (blueprint != null) // found!
            return blueprint;

        // Did not find a direct match, try matching to a blueprint by moduleCompleteKey
        blueprint = contentBlueprintManager.getById(id);

        if (blueprint == null) // UUID did not match to any content blueprints :(
        {
            log.warn("blueprint not found with id: " + id);
            return null;
        }

        blueprint = getBlueprintFromListByKey(blueprint, contentBlueprints);

        if (blueprint == null) // moduleCompleteKey did not match to any blueprint in the passed in list :(
        {
            log.warn("blueprint not found with id: " + id);
            return null;
        }

        return blueprint;
    }

    private ContentBlueprint getBlueprintFromListById(@Nonnull UUID uuid,
                                                      @Nonnull Collection<ContentBlueprint> contentBlueprints) {
        for (ContentBlueprint contentBlueprint : contentBlueprints) {
            if (uuid.equals(contentBlueprint.getId()))
                return contentBlueprint;
        }

        return null;
    }

    private ContentBlueprint getBlueprintFromListByKey(@Nonnull ContentBlueprint blueprint,
                                                       @Nonnull Collection<ContentBlueprint> contentBlueprints) {
        String blueprintModuleCompleteKey = blueprint.getModuleCompleteKey();

        for (ContentBlueprint contentBlueprint : contentBlueprints) {
            if (blueprintModuleCompleteKey.equals(contentBlueprint.getModuleCompleteKey()))
                return contentBlueprint; // return customised version of the promoted bp.
        }

        return null; // promoted bp is not matched to anything in the list so don't return it.
    }

    private List<UUID> convertStringToUuid(@Nonnull List<String> stringIds) {
        List<UUID> uuids = Lists.newArrayList();

        for (String id : stringIds) {
            uuids.add(UUID.fromString(id));
        }

        return uuids;
    }
}
