package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.content.template.ContentBlueprintInstance;
import com.atlassian.confluence.api.model.content.template.ContentTemplate;
import com.atlassian.confluence.api.model.content.template.ContentTemplateId;
import com.atlassian.confluence.api.model.content.template.ContentTemplateType;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.service.content.template.ContentTemplateService;
import com.atlassian.confluence.api.service.exceptions.BadRequestException;
import com.atlassian.confluence.api.service.exceptions.unchecked.NotImplementedServiceException;
import com.atlassian.confluence.plugins.createcontent.api.services.BlueprintContentTemplateService;
import com.atlassian.confluence.plugins.createcontent.api.services.PageContentTemplateService;
import com.atlassian.confluence.plugins.createcontent.factory.TemplateFinderFactory;
import com.atlassian.fugue.Option;

/**
 * CRA-1230 Facade service to delegate call to either Page or Blueprint ContentTemplateService
 */
public class DefaultContentTemplateService implements ContentTemplateService {
    private final PageContentTemplateService pageContentTemplateService;
    private final BlueprintContentTemplateService blueprintContentTemplateService;
    private final TemplateFinderFactory templateFinderFactory;

    public DefaultContentTemplateService(PageContentTemplateService pageContentTemplateService,
                                         BlueprintContentTemplateService blueprintContentTemplateService,
                                         TemplateFinderFactory templateFinderFactory) {
        this.pageContentTemplateService = pageContentTemplateService;
        this.blueprintContentTemplateService = blueprintContentTemplateService;
        this.templateFinderFactory = templateFinderFactory;
    }

    @Override
    public PageResponse<ContentTemplate> getTemplates(ContentTemplateType contentTemplateType, Option<Space> space, PageRequest pageRequest, Expansion... expansions) {
        PageResponse<ContentTemplate> result;
        if (contentTemplateType.equals(ContentTemplateType.BLUEPRINT)) {
            result = blueprintContentTemplateService.getTemplates(contentTemplateType, space, pageRequest, expansions);
        } else if (contentTemplateType.equals(ContentTemplateType.PAGE)) {
            result = pageContentTemplateService.getTemplates(contentTemplateType, space, pageRequest, expansions);
        } else {
            throw new NotImplementedServiceException("Unknown template type: " + contentTemplateType);
        }

        return result;
    }

    public PageResponse<ContentTemplate> getTemplates(Option<Space> space, PageRequest pageRequest, Expansion... expansions) {
        throw new NotImplementedServiceException("This method has been deprecated on interface, will remove.");
    }

    @Override
    public ContentTemplate getTemplate(ContentTemplateId contentTemplateId, Expansion... expansions) {
        if (contentTemplateId instanceof ContentTemplateId.ContentTemplateIdWithId) {
            return pageContentTemplateService.getTemplate(contentTemplateId, expansions);
        } else if (contentTemplateId instanceof ContentTemplateId.ContentTemplateIdWithUUID ||
                contentTemplateId instanceof ContentTemplateId.ContentTemplateIdWithKeys) {
            return blueprintContentTemplateService.getTemplate(contentTemplateId, expansions);
        }

        throw new BadRequestException("Expect an id (long), (UUID) or (\"ModuleCompleteKey@SpaceKey\"), but received: " + contentTemplateId);
    }

    @Override
    public ContentTemplate create(ContentTemplate contentTemplate, Expansion... expansions) {
        try {
            if (contentTemplate.getTemplateType().equals(ContentTemplateType.PAGE)) {
                return pageContentTemplateService.create(contentTemplate, expansions);
            } else if (contentTemplate.getTemplateType().equals(ContentTemplateType.BLUEPRINT)) {
                return blueprintContentTemplateService.create(contentTemplate, expansions);
            }
        } catch (Exception e) {
            throw new BadRequestException(e);
        }

        throw new BadRequestException("Unsupported template: " + contentTemplate);
    }

    @Override
    public ContentTemplate update(ContentTemplate contentTemplate, Expansion... expansions) {
        return pageContentTemplateService.update(contentTemplate, expansions);
    }

    @Override
    public void delete(ContentTemplateId contentTemplateId) {
        pageContentTemplateService.delete(contentTemplateId);
    }

    public Validator validator() {
        return null;
    }

    @Override
    public Validator validator(ContentTemplateType contentTemplateType) {
        if (contentTemplateType.equals(ContentTemplateType.PAGE)) {
            return pageContentTemplateService.validator(contentTemplateType);
        } else if (contentTemplateType.equals(ContentTemplateType.BLUEPRINT)) {
            return blueprintContentTemplateService.validator(contentTemplateType);
        }

        throw new BadRequestException("Unsupported template type: " + contentTemplateType);
    }

    @Override
    public TemplateFinder find(Expansion... expansions) {
        return templateFinderFactory.createFinder(this, expansions);
    }

    @Override
    public ContentBlueprintInstance createInstance(ContentBlueprintInstance blueprintInstance, Expansion... expansions) {
        return pageContentTemplateService.createInstance(blueprintInstance, expansions);
    }
}
