package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Content;
import com.atlassian.confluence.api.model.content.ContentStatus;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.content.template.ContentBlueprintId;
import com.atlassian.confluence.api.model.content.template.ContentBlueprintInstance;
import com.atlassian.confluence.api.model.content.template.ContentBlueprintSpec;
import com.atlassian.confluence.api.model.content.template.ContentTemplateId;
import com.atlassian.confluence.api.service.content.ContentService;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.pages.DraftsTransitionHelper;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateBlueprintPageRestEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.BlueprintPage;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageEntity;
import com.atlassian.fugue.Option;

import java.util.Map;

/**
 * This is an adapter class to bridge new CBI with older model objects used in ContentBlueprintService.
 * Converts a ContentBlueprintInstance into a CreateBlueprintPageEntity.
 * Converts page/draft to ContentBlueprintInstance for rest response.
 */
public class ContentBlueprintInstanceAdapter {
    private final ContentService contentService;
    private final DraftsTransitionHelper draftsTransitionHelper;

    public ContentBlueprintInstanceAdapter(ContentService contentService,
                                           DraftsTransitionHelper draftsTransitionHelper) {
        this.contentService = contentService;
        this.draftsTransitionHelper = draftsTransitionHelper;
    }

    CreateBlueprintPageEntity convertToEntity(ContentBlueprintInstance cbi) {
        Content contentIn = cbi.getContent();
        ContentBlueprintSpec spec = cbi.getContentBlueprintSpec();

        Space space = contentIn.getSpace();
        String spaceKey = space != null ? space.getKey() : null;
        String contentBlueprintId = getContentBlueprintUuid(spec.getBlueprintId());
        String moduleCompleteKey = getContentBlueprintKey(spec.getBlueprintId());
        String contentTemplateId = getContentTemplateId(spec.getContentTemplateId());
        String contentTemplateKey = getContentTemplateKey(spec.getContentTemplateId());
        String viewPermissionUsers = getViewPermissionUsers(spec.getContext());
        long parentPageId = getParentPageId(contentIn);

        return new CreateBlueprintPageRestEntity(spaceKey,
                contentBlueprintId,
                contentTemplateId,
                contentTemplateKey,
                contentIn.getTitle(),
                viewPermissionUsers,
                parentPageId,
                moduleCompleteKey,
                spec.getContext());
    }

    ContentBlueprintInstance convertToInstance(ContentEntityObject draft,
                                               ContentBlueprintInstance contentBlueprintInstance,
                                               Expansion[] expansions) {
        Content createdContent;
        if (draftsTransitionHelper.isSharedDraftsFeatureEnabled(contentBlueprintInstance.getContent().getSpace().getKey())) {
            createdContent = contentService.find(expansions)
                    .withStatus(ContentStatus.DRAFT)
                    .withId(draft.getContentId())
                    .fetchOneOrNull();
        } else {
            createdContent = Content.builder()
                    .id(ContentId.deserialise(draft.getIdAsString()))
                    .status(ContentStatus.DRAFT)
                    .title(draft.getTitle())
                    .build();
        }

        return ContentBlueprintInstance.builder()
                .content(createdContent)
                .contentBlueprintSpec(contentBlueprintInstance.getContentBlueprintSpec())
                .build();
    }

    ContentBlueprintInstance convertToInstance(BlueprintPage page,
                                               ContentBlueprintInstance contentBlueprintInstance,
                                               Expansion[] expansions) {
        Content createdContent = contentService.find(expansions).withId(page.getPage().getContentId()).fetchOneOrNull();

        return ContentBlueprintInstance.builder()
                .content(createdContent)
                .contentBlueprintSpec(contentBlueprintInstance.getContentBlueprintSpec())
                .build();
    }

    private long getParentPageId(Content contentIn) {
        ContentId parentPageId = contentIn.getParentId();
        return parentPageId.isSet() ? parentPageId.asLong() : 0;
    }

    private String getViewPermissionUsers(Map<String, Object> context) {
        return (String) context.get(ContentBlueprintSpec.VIEW_PERMISSION_USERS);
    }

    private String getContentTemplateId(Option<ContentTemplateId> contentTemplateId) {
        if (contentTemplateId.isEmpty())
            return null;

        if (contentTemplateId.get() instanceof ContentTemplateId.ContentTemplateIdWithId) {
            long id = ((ContentTemplateId.ContentTemplateIdWithId) contentTemplateId.get()).getId();
            return String.valueOf(id);
        }
        return null;
    }

    private String getContentTemplateKey(Option<ContentTemplateId> contentTemplateId) {
        if (contentTemplateId.isEmpty())
            return null;

        if (contentTemplateId.get() instanceof ContentTemplateId.ContentTemplateIdWithKeys)
            return ((ContentTemplateId.ContentTemplateIdWithKeys) contentTemplateId.get()).getModuleCompleteKey();

        return null;
    }

    private String getContentBlueprintKey(ContentBlueprintId blueprintId) {
        if (blueprintId instanceof ContentBlueprintId.ContentBlueprintIdWithKeys)
            return ((ContentBlueprintId.ContentBlueprintIdWithKeys) blueprintId).getModuleCompleteKey();

        return null;
    }

    private String getContentBlueprintUuid(ContentBlueprintId blueprintId) {
        if (blueprintId instanceof ContentBlueprintId.ContentBlueprintIdWithId)
            return ((ContentBlueprintId.ContentBlueprintIdWithId) blueprintId).getId();

        return null;
    }
}
