package com.atlassian.confluence.plugins.createcontent.concurrent;

import com.atlassian.beehive.ClusterLock;
import com.atlassian.beehive.ClusterLockService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterConcurrentLazyInsertExecutor implements LazyInsertExecutor {
    private static final Logger log = LoggerFactory.getLogger(ClusterConcurrentLazyInsertExecutor.class);

    private final ClusterLockService lockService;

    public ClusterConcurrentLazyInsertExecutor(ClusterLockService lockService) {
        this.lockService = lockService;
    }

    /**
     * Try read existing data or insert new data if read returns null, in a way that is concurrent-safe in cluster
     *
     * @param lockKey the key for acquiring lock. Different keys result in different locks (no contention).
     * @return either value returned by inserter#read() or value returned by inserter#insert()
     */
    public <T> T lazyInsertAndRead(LazyInserter<T> inserter, String lockKey) {
        T result = inserter.read();
        if (result != null) {
            return result;
        }

        ClusterLock lock = lockService.getLockForName(lockKey);
        lock.lock();
        try {
            result = inserter.read();
            if (result != null) {
                log.debug("Read empty, but some other threads/instances inserted data while waiting for lock.");
                return result;
            } else {
                log.debug("Read empty, re-read empty and insert data inside a lock.");
                return inserter.insert();
            }
        } finally {
            lock.unlock();
        }
    }
}
