package com.atlassian.confluence.plugins.createcontent;

import com.atlassian.confluence.plugins.createcontent.model.BlueprintState;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.ConfluenceUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

/**
 * <p>Manages the state (enabled/disabled) of Content Blueprint content blueprint plugin modules on a per space basis.
 * This is similar to {@link com.atlassian.plugin.PluginController} which manages this state on a global level.</p>
 */
public interface BlueprintStateController {
    /**
     * Enable a content blueprint plugin id for a particular space, or globally if space is null.
     *
     * @param blueprintId id of the blueprint to be enabled
     * @param space       the space to enable this module in
     */
    void enableBlueprint(UUID blueprintId, Space space);

    /**
     * Disable a content blueprint plugin by its id for a particular space, or globally if space is null.
     *
     * @param blueprintId id of the blueprint to be disabled
     * @param space       the space to disable this module in
     */
    void disableBlueprint(UUID blueprintId, Space space);

    /**
     * Disable a set of content blueprint plugin ids for a particular space, or globally if space is null.
     *
     * @param blueprintIds set of blueprint ids to be disabled
     * @param space        the space to disable this module in
     */
    void disableBlueprints(Set<UUID> blueprintIds, Space space);

    /**
     * Returns a set of disabled blueprints UUIDs.
     *
     * @param space (required) the space
     * @return all keys of disabled content blueprint plugin modules in the specified space, an empty set if none is found
     */
    Set<UUID> getDisabledBlueprintIds(Space space);

    /**
     * Returns a set of disabled blueprints module complete keys.
     *
     * @param space (required) the space
     * @return all keys of disabled content blueprint plugin modules in the specified space, an empty set if none is found
     */
    Set<String> getDisabledBlueprintModuleCompleteKeys(Space space);

    /**
     * Returns a set of blueprint keys that are available to a user in a given space. If a blueprint is content blueprint plugin backed,
     * it is only returned if its Condition evaluates to true and if the content blueprint plugin module is enabled.
     *
     * @param user  (optional) the user. null for anonymous
     * @param space (optional) the space. If null is specified, disabled in space state will be false for all results
     * @return all the keys of blueprints that are available to a user in the space, an empty set if none is found
     * @since 2.1.16
     */
    Map<UUID, BlueprintState> getAllContentBlueprintState(@Nonnull String section, @Nullable ConfluenceUser user, @Nullable Space space);
}
