package com.atlassian.confluence.plugins.createcontent.template;

import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PluginTemplateReference;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.plugins.createcontent.impl.I18nPageTemplate;
import com.atlassian.confluence.spaces.Space;

import java.util.List;

/**
 * In essence this helper is responsible for serving up PageTemplate's stored inside plugins.
 * <p>
 * When a plugin page template is edited, a copy of the template is inserted into the database. This copy is returned in place of the original.
 * This helper encapsulates this logic.
 * <p>
 * Calling this a "helper" so as not to introduce another manager that can cause confusion with {@link com.atlassian.confluence.pages.templates.PageTemplateManager}.
 * This interface is purely concerned with template retrieval (for now and into the future) so it does not make sense for it to implement {@link com.atlassian.confluence.pages.templates.PageTemplateManager}
 * which has write methods (and be vulnerable to API breakages).
 */
public interface PluginPageTemplateHelper {
    /**
     * @param pluginTemplateReference a reference identifying the plugin template
     * @return the customized page template if it exists, else the default template served directly from the plugin
     */
    PageTemplate getPageTemplate(PluginTemplateReference pluginTemplateReference);

    PageTemplate getPageTemplate(ContentTemplateRef contentTemplateRef);

    /**
     * <p>Retrieves all templates declared inside plugins.</p>
     * <p>
     * <p>Additionally, this method will load all "customisations" of plugin templates that are stored in the database against:</p>
     * <ul>
     * <li>a space (if a space is specified)</li>
     * <li>globally (if null is passed in for the space)</li>
     * </ul>
     * <p>
     * <p>Any customised templates on the space or global level will be returned "instead of" the original plugin template.</p>
     *
     * @param space the space (optional)
     * @return a list of plugin PageTemplate(s). Any modified templates in the specified will be returned in place of the default template.
     */
    List<PageTemplate> getPageTemplates(Space space);

    /**
     * <p>Retrieves all templates declared inside system plugins</p>
     * <p>
     * <p>Additionally, this method will load all "customisations" of system plugin templates that are globally stored in the database</p>
     * <p>
     * <p>Any customised templates on the space or global level will be returned "instead of" the original plugin template.</p>
     */
    List<I18nPageTemplate> getSystemPageTemplates();
}
