package com.atlassian.confluence.plugins.createcontent.services;

import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.plugin.web.descriptors.WebItemModuleDescriptor;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.UUID;

/**
 * Finds Blueprints and Blueprint-related entities.
 * <p>
 * This interface stands between the publicly-exposed Services and the DB-gnostic Managers, so that neither of those
 * two need to care about how Blueprints are retrieved.
 *
 * @since 2.1
 */
public interface BlueprintResolver {
    WebItemModuleDescriptor getWebItemMatchingBlueprint(String blueprintModuleKey);

    WebItemModuleDescriptor getWebItemMatchingBlueprint(UUID blueprintId);

    /**
     * Given a template ref, finds the "nearest" template ref - space, global or plugin
     * that should be used when editing the template or creating a page from it.
     * <p>
     * This resolution is required because we might get a template ref from a space-override of a Blueprint,
     * but that template has been reverted. In that case, a globally-customised template ref for that Blueprint
     * should be used, if it exists. If not, the template ref from the original plugin clone Blueprint.
     */
    ContentTemplateRef resolveTemplateRef(ContentTemplateRef templateRef);

    /**
     * Given a Blueprint Id, finds the "nearest" Blueprint - space, global or plugin (in that order) that should be
     * used by calling code.
     * <p>
     * This resolution is required because we might get a Blueprint Id from storage, but that Blueprint has been edited
     * at the space or global level.
     * <p>
     * Note that this method does not allow for Ids that don't resolve to a Blueprint, it throws an exception rather
     * than returning null.
     */
    @Nonnull
    ContentBlueprint resolveContentBlueprint(@Nonnull String blueprintId, @Nullable String spaceKey)
            throws IllegalArgumentException;

    /**
     * Finds the correct Blueprint for a given context.
     */
    @Nonnull
    ContentBlueprint getContentBlueprint(String contentBlueprintId, String blueprintModuleCompleteKey, String spaceKey)
            throws IllegalArgumentException;
}
