package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.bandana.BandanaManager;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.plugins.createcontent.SpaceBandanaContext;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.services.PromotedTemplateService;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import static com.atlassian.confluence.plugins.createcontent.rest.BlueprintStateResource.NON_SPACE_ADMIN_PERMISSION_DENIED_RESPONSE;

@Component
public class DefaultPromotedTemplateService implements PromotedTemplateService {
    public static final String KEY_PROMOTED_TEMPLATES = "promotedTemplates";

    private final SpaceManager spaceManager;
    private final PermissionManager permissionManager;
    private final BandanaManager bandanaManager;

    @Autowired
    public DefaultPromotedTemplateService(
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport PermissionManager permissionManager,
            final @ComponentImport BandanaManager bandanaManager) {
        this.spaceManager = spaceManager;
        this.permissionManager = permissionManager;
        this.bandanaManager = bandanaManager;
    }


    @Override
    public void promoteTemplate(long templateId, @Nonnull String spaceKey)
            throws BlueprintIllegalArgumentException {
        toggleTemplatePromotion(templateId, spaceKey, true);
    }

    @Override
    public void demoteTemplate(long templateId, @Nonnull String spaceKey)
            throws BlueprintIllegalArgumentException {
        toggleTemplatePromotion(templateId, spaceKey, false);
    }

    @Override
    public List<Long> getPromotedTemplates(@Nonnull Space space) {
        // get promoted template ids
        final List<Long> promotedTemplateIds = (List<Long>) bandanaManager.getValue(new SpaceBandanaContext(space), KEY_PROMOTED_TEMPLATES);
        if (promotedTemplateIds == null) // no promoted templates
            return Lists.newArrayList();

        return promotedTemplateIds;
    }

    private void toggleTemplatePromotion(long templateId, @Nonnull String spaceKey, boolean promote)
            throws BlueprintIllegalArgumentException {
        if (StringUtils.isBlank(spaceKey))
            throw new BlueprintIllegalArgumentException("Space key is required to promote/demote template with id: " + templateId, ResourceErrorType.PARAMETER_MISSING, "spaceKey");

        final Space space = spaceManager.getSpace(spaceKey);

        if (space == null)
            throw new BlueprintIllegalArgumentException("Space with key '" + spaceKey + "' could not be found.", ResourceErrorType.NOT_FOUND_SPACE, spaceKey);

        if (!permissionManager.hasPermission(AuthenticatedUserThreadLocal.get(), Permission.ADMINISTER, space))
            throw new BlueprintIllegalArgumentException(NON_SPACE_ADMIN_PERMISSION_DENIED_RESPONSE, ResourceErrorType.PERMISSION_USER_ADMIN_SPACE, spaceKey);

        SpaceBandanaContext spaceBandanaContext = new SpaceBandanaContext(space);
        List<Long> ids = (List<Long>) bandanaManager.getValue(spaceBandanaContext, KEY_PROMOTED_TEMPLATES);
        Set<Long> promotedTemplatesIds = Sets.newHashSet();

        if (ids != null)
            promotedTemplatesIds.addAll(ids);

        if (promote)
            promotedTemplatesIds.add(templateId);
        else
            promotedTemplatesIds.remove(templateId);

        bandanaManager.setValue(spaceBandanaContext, KEY_PROMOTED_TEMPLATES, Lists.newArrayList(promotedTemplatesIds));
    }

    private long matchTemplateId(Long promotedTemplateId, Collection<PageTemplate> templatesInSpace) {
        // Try match templateId to templates in list
        for (PageTemplate templateInSpace : templatesInSpace) {
            long id = templateInSpace.getId();
            if (promotedTemplateId.equals(id))
                return promotedTemplateId;
        }

        return 0;
    }

}
