package com.atlassian.confluence.plugins.createcontent.factory;

import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.service.exceptions.NotFoundException;
import com.atlassian.confluence.api.service.finder.ManyFetcher;
import com.atlassian.confluence.api.service.finder.SingleFetcher;
import com.atlassian.confluence.plugins.createcontent.transaction.ThrowingTransactionCallback;
import com.atlassian.confluence.plugins.createcontent.transaction.ThrowingTransactionTemplate;
import com.atlassian.confluence.spring.transaction.interceptor.TransactionalHostContextAccessor;

import java.util.Optional;

/**
 * Wraps all the {@link SingleFetcher} and
 * {@link ManyFetcher} methods in a
 * transaction template, so that we ensure all results are returned in (at least) the same read only
 * transaction.
 * TODO: move this to core, so ContentProperty and CreateContent can share the same class.
 */
public abstract class TransactionWrappingFinder<T> implements SingleFetcher<T>, ManyFetcher<T> {
    private final ThrowingTransactionTemplate transactionTemplate;
    private final SingleFetcher<T> singleFetcherDelegate;
    private final ManyFetcher<T> manyFetcherDelegate;

    public TransactionWrappingFinder(SingleFetcher<T> singleFetcherDelegate, ManyFetcher<T> manyFetcherDelegate, TransactionalHostContextAccessor hostContextAccessor) {
        this.singleFetcherDelegate = singleFetcherDelegate;
        this.manyFetcherDelegate = manyFetcherDelegate;
        this.transactionTemplate = new ThrowingTransactionTemplate(hostContextAccessor);
    }

    @Override
    public PageResponse<T> fetchMany(final PageRequest request) throws NotFoundException {
        ThrowingTransactionCallback<PageResponse<T>, NotFoundException> callback =
                () -> manyFetcherDelegate.fetchMany(request);
        return executeReadOnly(callback);
    }

    private <T> T executeReadOnly(
            ThrowingTransactionCallback<T, NotFoundException> callback)
            throws NotFoundException {
        return transactionTemplate.execute(NotFoundException.class,
                TransactionalHostContextAccessor.Permission.READ_ONLY,
                callback);
    }

    @Override
    public com.atlassian.fugue.Option<T> fetchOne() {
        ThrowingTransactionCallback<com.atlassian.fugue.Option<T>, NotFoundException> callback =
                singleFetcherDelegate::fetchOne;
        return executeReadOnly(callback);
    }

    @Override
    public T fetchOneOrNull() {
        return fetchOne().getOrNull();
    }

    @Override
    public Optional<T> fetch() {
        ThrowingTransactionCallback<Optional<T>, NotFoundException> callback =
                singleFetcherDelegate::fetch;
        return executeReadOnly(callback);
    }
}
