package com.atlassian.confluence.plugins.createcontent;

import com.atlassian.confluence.content.render.xhtml.ConversionContext;
import com.atlassian.confluence.macro.Macro;
import com.atlassian.confluence.macro.MacroExecutionException;
import com.atlassian.confluence.plugins.createcontent.api.services.CreateButtonService;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;

import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

public class CreateFromTemplateMacro implements Macro {
    private static final String SPACE_KEY = "spaceKey";
    private static final String TEMPLATE_ID = "templateId";
    private static final String BLUEPRINT_MODULE_COMPLETE_KEY = "blueprintModuleCompleteKey";
    private static final String NEW_PAGE_TITLE = "title";
    private static final String CONTENT_BLUEPRINT_ID = "contentBlueprintId";
    private static final String OLD_BUTTON_LABEL = "createButtonLabel"; // some of our bundled blueprints still use this
    private static final String BUTTON_LABEL = "buttonLabel";

    private final SpaceManager spaceManager;
    private final CreateButtonService createButtonService;

    public CreateFromTemplateMacro(SpaceManager spaceManager, CreateButtonService createButtonService) {
        this.spaceManager = spaceManager;
        this.createButtonService = createButtonService;
    }

    @Override
    public String execute(Map<String, String> params, String body, ConversionContext conversionContext) throws MacroExecutionException {
        Space space = getSpace(params, conversionContext);
        String title = params.get(NEW_PAGE_TITLE);
        String buttonLabelKey = params.get(BUTTON_LABEL);
        if (buttonLabelKey == null)
            buttonLabelKey = params.get(OLD_BUTTON_LABEL);


        String templateId = params.get(TEMPLATE_ID);
        if (isNotBlank(templateId)) {
            return createButtonService.renderTemplateButton(space, Long.parseLong(templateId), buttonLabelKey, title);
        }

        String contentBlueprintId = params.get(CONTENT_BLUEPRINT_ID);
        String blueprintModuleCompleteKey = params.get(BLUEPRINT_MODULE_COMPLETE_KEY);
        return createButtonService.renderBlueprintButton(space, contentBlueprintId, blueprintModuleCompleteKey, buttonLabelKey, title);
    }

    private Space getSpace(Map<String, String> params, ConversionContext conversionContext) throws MacroExecutionException {
        String spaceKey = params.get(SPACE_KEY);
        if (isBlank(spaceKey))
            spaceKey = conversionContext.getSpaceKey();

        Space space = spaceManager.getSpace(spaceKey);
        if (space == null)
            throw new MacroExecutionException("No space found with space key: " + spaceKey);

        return space;
    }

    @Override
    public BodyType getBodyType() {
        return BodyType.NONE;
    }

    @Override
    public OutputType getOutputType() {
        return OutputType.INLINE;
    }
}
