package com.atlassian.confluence.plugins.createcontent.upgrade;

import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.plugins.createcontent.BlueprintConstants;
import com.atlassian.confluence.plugins.createcontent.activeobjects.ContentBlueprintAo;
import com.atlassian.confluence.plugins.createcontent.activeobjects.ContentTemplateRefAo;
import com.atlassian.confluence.plugins.createcontent.exceptions.BlueprintPluginNotFoundException;
import com.atlassian.confluence.plugins.createcontent.listeners.TemplateUpdatedListener;
import com.atlassian.confluence.plugins.createcontent.services.TemplateUpdater;
import com.atlassian.confluence.plugins.createcontent.template.PluginPageTemplateHelper;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.sal.api.message.Message;
import com.atlassian.sal.api.upgrade.PluginUpgradeTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;

/**
 * Upgrades the content blueprint reference in ContentBlueprint AO table for user-edited blueprint templates.
 * {@link ContentBlueprintAo} keeps a reference to a list of {@link ContentTemplateRefAo} used when an user creates a
 * page from a blueprint. By default, the one that comes bundled with the plugin is used. If the user edits it, {@link
 * TemplateUpdatedListener} updates the content template reference in AO. If the template was already modified before
 * upgrading Confluence, we need to update the content template reference in AO to use the user-defined one.
 */
@Component
@ExportAsService(PluginUpgradeTask.class)
public class SpaceEditedTemplatesUpgradeTask implements PluginUpgradeTask {
    private static final Logger log = LoggerFactory.getLogger(SpaceEditedTemplatesUpgradeTask.class);

    private final SpaceManager spaceManager;
    private final PluginPageTemplateHelper pluginPageTemplateHelper;
    private final TemplateUpdater templateUpdater;

    @Autowired
    public SpaceEditedTemplatesUpgradeTask(SpaceManager spaceManager, PluginPageTemplateHelper pluginPageTemplateHelper,
                                           TemplateUpdater templateUpdater) {
        this.spaceManager = spaceManager;
        this.pluginPageTemplateHelper = pluginPageTemplateHelper;
        this.templateUpdater = templateUpdater;
    }

    /**
     * The build number for this upgrade task. Once this upgrade task has run the plugin manager will store this build
     * number against this plugin type.  After this only upgrade tasks with higher build numbers will be run
     */
    @Override
    public int getBuildNumber() {
        return 4;
    }

    @Override
    public String getShortDescription() {
        return "Updates the content template reference of the content blueprint for user-edited templates.";
    }

    @Override
    public Collection<Message> doUpgrade() {
        List<Space> allSpaces = spaceManager.getAllSpaces();

        for (Space space : allSpaces) {
            // Get all space-level templates declared inside a plugin
            List<PageTemplate> templates = pluginPageTemplateHelper.getPageTemplates(space);
            updateBlueprintContentTemplates(templates);
        }

        return null;
    }

    // Upgrades the content template referenced if it belongs to a blueprint.
    private void updateBlueprintContentTemplates(List<PageTemplate> pageTemplates) {
        for (PageTemplate pageTemplate : pageTemplates) {
            try {
                templateUpdater.updateContentTemplateRef(pageTemplate);
            } catch (BlueprintPluginNotFoundException e) {
                log.warn(
                        "Page template '{}' (ID: {}) could not be migrated because a plugin was disabled. Cause: {}",
                        pageTemplate.getTitle(),
                        pageTemplate.getId(),
                        e.getMessage()
                );
            }
        }
    }

    @Override
    public String getPluginKey() {
        return BlueprintConstants.PLUGIN_KEY;
    }
}
