package com.atlassian.confluence.plugins.createcontent.transaction;

import com.atlassian.confluence.spring.transaction.interceptor.TransactionalHostContextAccessor;
import com.google.common.base.Throwables;
import com.google.common.util.concurrent.UncheckedExecutionException;

import static com.atlassian.confluence.spring.transaction.interceptor.TransactionalHostContextAccessor.Permission;
import static com.atlassian.confluence.spring.transaction.interceptor.TransactionalHostContextAccessor.Propagation;

/**
 * TransactionTemplate that supports throwing checked exceptions.
 *
 * @since 5.6
 */
public class ThrowingTransactionTemplate {
    private final TransactionalHostContextAccessor hostContextAccessor;

    public ThrowingTransactionTemplate(TransactionalHostContextAccessor hostContextAccessor) {
        this.hostContextAccessor = hostContextAccessor;
    }

    /**
     * Runs an action in a transaction and returns a optional value.
     *
     * @param propagation the propagation mode to use
     * @param callback    The callback class to execute
     * @return Optional result of the operation. May be null
     * @throws X                if it was thrown by the callback
     * @throws RuntimeException if anything went wrong.  The caller will be responsible for rolling back.
     */
    public <T, X extends Exception> T execute(Class<X> exceptionType,
                                              Propagation propagation,
                                              ThrowingTransactionCallback<T, X> callback)
            throws X {
        return execute(exceptionType, propagation, Permission.READ_ONLY, callback);
    }

    /**
     * Runs an action in a transaction and returns a optional value. The transaction is {@link Propagation#REQUIRED}.
     *
     * @param permission if {@link Permission#READ_ONLY}, attempt to use a read only transaction
     * @param callback   The callback class to execute
     * @return Optional result of the operation. May be null
     * @throws X                if it was thrown by the callback
     * @throws RuntimeException if anything went wrong.  The caller will be responsible for rolling back.
     */
    public <T, X extends Exception> T execute(Class<X> exceptionType,
                                              Permission permission,
                                              ThrowingTransactionCallback<T, X> callback)
            throws X {
        return execute(exceptionType, Propagation.REQUIRED, permission, callback);
    }

    /**
     * Runs an action in a transaction and returns a optional value.
     *
     * @param propagation the propagation mode to use
     * @param permission  if {@link Permission#READ_ONLY}, attempt to use a read only transaction
     * @param callback    The callback class to execute
     * @return Optional result of the operation. May be null
     * @throws X                if it was thrown by the callback
     * @throws RuntimeException if anything went wrong.  The caller will be responsible for rolling back.
     */
    public <T, X extends Exception> T execute(final Class<X> exceptionType,
                                              Propagation propagation,
                                              Permission permission,
                                              final ThrowingTransactionCallback<T, X> callback)
            throws X {
        try {
            return hostContextAccessor.doInTransaction(propagation, permission, () -> {
                try {
                    return callback.doInTransaction();
                } catch (Exception e) {
                    if (exceptionType.isInstance(e))
                        throw new UncheckedExecutionException(e);
                    Throwables.propagateIfPossible(e);
                }
                return null;
            });
        } catch (UncheckedExecutionException e) {
            if (exceptionType.isInstance(e.getCause()))
                throw exceptionType.cast(e.getCause());
            Throwables.propagateIfPossible(e.getCause());
            throw e;
        }
    }

    /**
     * Runs an action in a transaction and returns a optional value.
     *
     * @param propagation the propagation mode to use
     * @param callback    The callback class to execute
     * @return Optional result of the operation. May be null
     * @throws X1               if it was thrown by the callback
     * @throws X2               if it was thrown by the callback
     * @throws RuntimeException if anything went wrong.  The caller will be responsible for rolling back.
     */
    public <T, X1 extends Exception, X2 extends Exception> T execute(Class<X1> exceptionType,
                                                                     Class<X2> exceptionType2,
                                                                     Propagation propagation,
                                                                     Throwing2TransactionCallback<T, X1, X2> callback)
            throws X1, X2 {
        return execute(exceptionType, exceptionType2, propagation, Permission.READ_ONLY, callback);
    }

    /**
     * Runs an action in a transaction and returns a optional value. The transaction is {@link Propagation#REQUIRED}.
     *
     * @param permission if {@link Permission#READ_ONLY}, attempt to use a read only transaction
     * @param callback   The callback class to execute
     * @return Optional result of the operation. May be null
     * @throws X1               if it was thrown by the callback
     * @throws X2               if it was thrown by the callback
     * @throws RuntimeException if anything went wrong.  The caller will be responsible for rolling back.
     */
    public <T, X1 extends Exception, X2 extends Exception> T execute(Class<X1> exceptionType,
                                                                     Class<X2> exceptionType2,
                                                                     Permission permission,
                                                                     Throwing2TransactionCallback<T, X1, X2> callback)
            throws X1, X2 {
        return execute(exceptionType, exceptionType2, Propagation.REQUIRED, permission, callback);
    }

    /**
     * Runs an action in a transaction and returns a optional value.
     *
     * @param propagation the propagation mode to use
     * @param permission  if {@link Permission#READ_ONLY}, attempt to use a read only transaction
     * @param callback    The callback class to execute
     * @return Optional result of the operation. May be null
     * @throws X1               if it was thrown by the callback
     * @throws X2               if it was thrown by the callback
     * @throws RuntimeException if anything went wrong.  The caller will be responsible for rolling back.
     */
    public <T, X1 extends Exception, X2 extends Exception> T execute(final Class<X1> exceptionType,
                                                                     final Class<X2> exceptionType2,
                                                                     Propagation propagation,
                                                                     Permission permission,
                                                                     final Throwing2TransactionCallback<T, X1, X2> callback)
            throws X1, X2 {
        try {
            return hostContextAccessor.doInTransaction(propagation, permission, () -> {
                try {
                    return callback.doInTransaction();
                } catch (Exception e) {
                    if (exceptionType.isInstance(e))
                        throw new UncheckedExecutionException(e);
                    Throwables.propagateIfPossible(e);
                }
                return null;
            });
        } catch (UncheckedExecutionException e) {
            Throwables.propagateIfPossible(e.getCause(), exceptionType, exceptionType2);
            throw e;
        }
    }
}
