package com.atlassian.confluence.plugins.createcontent.template;

import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PageTemplateManager;
import com.atlassian.confluence.pages.templates.PluginTemplateReference;
import com.atlassian.confluence.plugins.createcontent.extensions.ContentTemplateModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.plugins.createcontent.impl.I18nPageTemplate;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.Sets.newHashSet;

@Component
public class DefaultPluginPageTemplateHelper implements PluginPageTemplateHelper {
    private static final Logger log = LoggerFactory.getLogger(DefaultPluginPageTemplateHelper.class);

    private final PageTemplateManager pageTemplateManager;
    private final PluginAccessor pluginAccessor;

    @Autowired
    public DefaultPluginPageTemplateHelper(
            @ComponentImport PageTemplateManager pageTemplateManager,
            @ComponentImport PluginAccessor pluginAccessor) {
        this.pageTemplateManager = pageTemplateManager;
        this.pluginAccessor = pluginAccessor;
    }

    @Override
    public PageTemplate getPageTemplate(PluginTemplateReference pluginTemplateReference) {
        // Look for space-level template overrides for this blueprint.
        PageTemplate result = pageTemplateManager.getPageTemplate(pluginTemplateReference);
        if (result != null)
            return result;

        // Look for site-level template overrides for this blueprint.
        result = pageTemplateManager.getPageTemplate(PluginTemplateReference.globalTemplateReference(
                pluginTemplateReference.getModuleCompleteKey(), pluginTemplateReference.getReferencingModuleCompleteKey()));
        if (result != null)
            return result;

        // Look for the original template bundled with the plugin
        PageTemplate pageTemplateByModuleKey = getPageTemplateByModuleKey(pluginTemplateReference.getModuleCompleteKey().getCompleteKey());
        if (pageTemplateByModuleKey == null)
            throw new IllegalStateException("No PageTemplate found matching reference: " + pluginTemplateReference);

        return pageTemplateByModuleKey;
    }

    @Override
    public PageTemplate getPageTemplate(ContentTemplateRef contentTemplateRef) {
        PageTemplate pageTemplate;
        long id = contentTemplateRef.getTemplateId();
        if (id != 0) {
            pageTemplate = pageTemplateManager.getPageTemplate(id);
        } else {
            // Look for the original template bundled with the plugin
            pageTemplate = getPageTemplateByModuleKey(contentTemplateRef.getModuleCompleteKey());
        }

        if (pageTemplate == null)
            throw new IllegalStateException("No PageTemplate found for ContentTemplateRef: " + contentTemplateRef);

        return pageTemplate;
    }

    private PageTemplate getPageTemplateByModuleKey(String moduleCompleteKey) {
        PageTemplate result = null;
        ModuleDescriptor contentTemplateModuleDescriptor = pluginAccessor.getEnabledPluginModule(moduleCompleteKey);

        if (contentTemplateModuleDescriptor instanceof ContentTemplateModuleDescriptor) {
            try {
                PageTemplate pluginPageTemplate = ((ContentTemplateModuleDescriptor) contentTemplateModuleDescriptor).getModule();

                if (pluginPageTemplate != null) {
                    /*
                      Make defensive copy of template so that clients cannot modify the original instance stored in the module descriptor
                      I don't have time now to introduce a read-only transfer object over PageTemplate's.
                     */
                    result = (PageTemplate) pluginPageTemplate.clone();
                }
            } catch (Exception e) {
                log.debug("Error constructing a PageTemplate instance from content template descriptor", e);
            }
        }

        return result;
    }

    @Override
    public List<PageTemplate> getPageTemplates(Space space) {
        ImmutableList.Builder<PageTemplate> result = ImmutableList.builder();
        Set<ModuleCompleteKey> moduleKeys = newHashSet();

        Collection<PageTemplate> pluginPageTemplates = pluginAccessor.getEnabledModulesByClass(PageTemplate.class);

        for (PageTemplate pluginPageTemplate : pluginPageTemplates) {
            moduleKeys.add(pluginPageTemplate.getModuleCompleteKey());
            result.add(pluginPageTemplate);
        }

        //noinspection unchecked
        List<PageTemplate> databasePageTemplates;
        if (space != null)
            databasePageTemplates = space.getPageTemplates();
        else
            databasePageTemplates = pageTemplateManager.getGlobalPageTemplates();
        for (PageTemplate pageTemplate : databasePageTemplates) {
            if (StringUtils.isBlank(pageTemplate.getPluginKey()))
                continue;

            ModuleCompleteKey moduleCompleteKey = pageTemplate.getModuleCompleteKey();
            if (moduleKeys.contains(moduleCompleteKey))
                result.add(pageTemplate); // override default template with one the user has customized
        }

        return result.build();
    }

    @Override
    public List<I18nPageTemplate> getSystemPageTemplates() {
        final Map<ModuleCompleteKey, I18nPageTemplate> result = new LinkedHashMap<>();

        final Collection<ContentTemplateModuleDescriptor> pluginPageTemplates = pluginAccessor.getEnabledModuleDescriptorsByClass(ContentTemplateModuleDescriptor.class);

        for (ContentTemplateModuleDescriptor pluginPageTemplate : pluginPageTemplates) {
            final String pluginKey = pluginPageTemplate.getPluginKey();
            if (pluginAccessor.isSystemPlugin(pluginKey)) {
                final PageTemplate module = pluginPageTemplate.getModule();
                result.put(module.getModuleCompleteKey(), new I18nPageTemplate(pluginPageTemplate.getNameKey(), module));
            }
        }

        final List<PageTemplate> databasePageTemplates = pageTemplateManager.getGlobalPageTemplates();

        for (PageTemplate pageTemplate : databasePageTemplates) {
            final String pluginKey = pageTemplate.getPluginKey();
            if (StringUtils.isBlank(pluginKey) || !pluginAccessor.isSystemPlugin(pluginKey))
                continue;

            final ModuleCompleteKey moduleCompleteKey = pageTemplate.getModuleCompleteKey();
            if (result.containsKey(moduleCompleteKey)) {
                I18nPageTemplate old = result.get(moduleCompleteKey);
                result.put(moduleCompleteKey, new I18nPageTemplate(old.getI18nNameKey(), pageTemplate));  // override default template with one the user has customized
            }
        }

        return ImmutableList.copyOf(result.values());
    }
}
