package com.atlassian.confluence.plugins.createcontent.services;

import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.spaces.Space;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;

public interface PromotedBlueprintService {

    /**
     * Takes in a list of blueprints and returns the ones that are promoted in the specified space.
     * Please note, the blueprints may be enabled or disabled in the context.
     * <p>
     * Filtering out enabled and disabled &lt;blueprint&gt;s are a separate concern.
     *
     * @return a collection of promoted blueprint entities.
     */
    @Nonnull
    Collection<ContentBlueprint> getPromotedBlueprints(@Nonnull Collection<ContentBlueprint> blueprints, @Nullable Space space);

    /**
     * Promote a blueprint.
     *
     * @param blueprintId blueprint UUID string
     * @param spaceKey    space in which the blueprint is to be promoted or demoted
     * @return true if the Blueprint was promoted (i.e. if it wasn't already promoted)
     */
    boolean promoteBlueprint(@Nonnull String blueprintId, @Nonnull String spaceKey) throws BlueprintIllegalArgumentException;

    /**
     * Unpromote a blueprint
     *
     * @param blueprintId blueprint UUID string
     * @param spaceKey    space in which the blueprint is to be promoted or demoted
     * @return true if the Blueprint was unpromoted (i.e. if it was previously promoted)
     */
    boolean demoteBlueprint(@Nonnull String blueprintId, @Nonnull String spaceKey) throws BlueprintIllegalArgumentException;

    /**
     * Promote a list of blueprints
     *
     * @param blueprintIds List of blueprint UUID strings
     * @param space        space in which the blueprints are to be promoted
     */
    void promoteBlueprints(@Nonnull List<String> blueprintIds, @Nonnull Space space);
}
