package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.api.service.accessmode.AccessModeService;
import com.atlassian.confluence.api.service.accessmode.ReadOnlyAccessAllowed;
import com.atlassian.confluence.api.service.accessmode.ReadOnlyAccessBlocked;
import com.atlassian.confluence.plugins.createcontent.conditions.ShowSpaceWelcomeDialogCondition;
import com.atlassian.confluence.plugins.createcontent.services.SpaceCollectionService;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.security.SpacePermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.core.AtlassianCoreException;
import com.atlassian.core.user.preferences.UserPreferences;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;
import org.slf4j.Logger;

import javax.ws.rs.Consumes;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import java.util.List;
import java.util.Map;

import static com.atlassian.confluence.security.PermissionManager.TARGET_APPLICATION;
import static com.atlassian.confluence.security.SpacePermission.ADMINISTER_SPACE_PERMISSION;
import static org.slf4j.LoggerFactory.getLogger;

@Path("/spaces")
public class SpaceResource extends AbstractRestResource {
    private static final Logger log = getLogger(SpaceResource.class);
    public static final String PARAM_PROMOTED_SPACE_KEY = "promotedSpaceKey";
    public static final String PARAM_PROMOTED_SPACES_LIMIT = "promotedSpacesLimit";
    public static final String PARAM_OTHER_SPACES_LIMIT = "otherSpacesLimit";
    public static final String PARAM_SPACE_KEY = "spaceKey";

    private final SpaceManager spaceManager;
    private final SpacePermissionManager spacePermissionManager;
    private final UserAccessor userAccessor;
    private final SpaceCollectionService spaceCollectionService;

    public SpaceResource(
            final @ComponentImport PermissionManager permissionManager,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport SpacePermissionManager spacePermissionManager,
            final @ComponentImport UserAccessor userAccessor,
            final SpaceCollectionService spaceCollectionService,
            final @ComponentImport AccessModeService accessModeService) {
        super(permissionManager, spaceManager, accessModeService);

        this.spaceManager = spaceManager;
        this.spacePermissionManager = spacePermissionManager;
        this.userAccessor = userAccessor;
        this.spaceCollectionService = spaceCollectionService;
    }

    @GET
    @ReadOnlyAccessBlocked
    @AnonymousAllowed
    public Map<String, SpaceResultsEntity> getSpaces(@QueryParam(PARAM_PROMOTED_SPACE_KEY) List<String> promotedSpaceKeys,
                                                     @QueryParam(PARAM_PROMOTED_SPACES_LIMIT) @DefaultValue("10") int promotedSpacesLimit,
                                                     @QueryParam(PARAM_OTHER_SPACES_LIMIT) @DefaultValue("10") int otherSpacesLimit) {
        return spaceCollectionService.getSpaces(promotedSpaceKeys, promotedSpacesLimit, otherSpacesLimit, null);
    }

    @GET
    @Path("/adminCheck")
    @AnonymousAllowed
    public boolean spaceAdministrationPermissionCheck(@QueryParam(PARAM_SPACE_KEY) String spaceKey) {
        checkEmptyParameter(spaceKey, PARAM_SPACE_KEY);

        ConfluenceUser user = getUser();
        return hasAdminPermission(spaceKey, user);
    }

    @POST
    @ReadOnlyAccessAllowed
    @Path("/skip-space-welcome-dialog")
    @Consumes({MediaType.APPLICATION_JSON})
    public void skipSpaceWelcomeDialog() {
        ConfluenceUser user = getUser();
        final UserPreferences userPreferences = userAccessor.getUserPreferences(user);
        try {
            userPreferences.setBoolean(ShowSpaceWelcomeDialogCondition.SPACE_WELCOME_DIALOG_DISMISSED_KEY, true);
        } catch (AtlassianCoreException e) {
            log.warn("Unable to skip space welcome dialog for user: {}", user);
        }
    }

    private boolean hasAdminPermission(String spaceKey, ConfluenceUser user) {
        if (user != null) {
            Space space = spaceManager.getSpace(spaceKey);
            if (space != null) {
                return spacePermissionManager.hasPermission(ADMINISTER_SPACE_PERMISSION, space, user);
            }
        }
        return false;
    }

    /**
     * @return true if the user can create a space
     */
    @GET
    @Path("can-create-spaces")
    @AnonymousAllowed
    public Boolean canCreateSpaces() {
        return permissionManager.hasCreatePermission(getUser(), TARGET_APPLICATION, com.atlassian.confluence.spaces.Space.class);
    }

}