package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.api.service.accessmode.AccessModeService;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.api.services.ContentBlueprintService;
import com.atlassian.confluence.plugins.createcontent.extensions.UserBlueprintConfigManager;
import com.atlassian.confluence.plugins.createcontent.rest.entities.BlueprintDraftEntity;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateBlueprintPageRestEntity;
import com.atlassian.confluence.plugins.createcontent.rest.entities.PageEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.BlueprintPage;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.xwork.FlashScope;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;

import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import java.util.UUID;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.APPLICATION_XML;


/**
 * For creating Pages from Content Blueprints.
 *
 * @since 2.0
 */
@Path("/content-blueprint")
public class ContentBlueprintResource extends AbstractRestResource {
    private final ContentBlueprintService contentBlueprintService;
    private final SettingsManager settingsManager;
    private final UserBlueprintConfigManager userBlueprintConfigManager;

    public ContentBlueprintResource(
            final @ComponentImport PermissionManager permissionManager,
            final @ComponentImport SpaceManager spaceManager,
            final ContentBlueprintService legacyContentBlueprintService,
            final @ComponentImport SettingsManager settingsManager,
            final UserBlueprintConfigManager userBlueprintConfigManager,
            final @ComponentImport AccessModeService accessModeService) {
        super(permissionManager, spaceManager, accessModeService);

        this.contentBlueprintService = legacyContentBlueprintService;
        this.settingsManager = settingsManager;
        this.userBlueprintConfigManager = userBlueprintConfigManager;
    }

    @POST
    @Path("create-content")
    @AnonymousAllowed
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public PageEntity createPage(final CreateBlueprintPageRestEntity entity) throws BlueprintIllegalArgumentException {
        checkNullEntity(entity);

        ConfluenceUser user = getUser();

        boolean firstBlueprintCreation = false;
        if (user != null) {
            firstBlueprintCreation = (entity.getContentBlueprintId() != null)
                    && userBlueprintConfigManager.isFirstBlueprintOfTypeForUser(UUID.fromString(entity.getContentBlueprintId()), user);
        }

        BlueprintPage page = contentBlueprintService.createPage(entity, user);

        String baseUrl = settingsManager.getGlobalSettings().getBaseUrl();
        PageEntity result = new PageEntity(page, baseUrl);

        if (firstBlueprintCreation) {
            enableFirstBlueprintCreationFlashScope(result);
        }

        return result;
    }

    @POST
    @Path("create-draft")
    @AnonymousAllowed
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public BlueprintDraftEntity createDraft(final CreateBlueprintPageRestEntity entity) throws BlueprintIllegalArgumentException {
        checkNullEntity(entity);

        ContentEntityObject contentDraft = contentBlueprintService.createContentDraft(entity, getUser());
        String baseUrl = settingsManager.getGlobalSettings().getBaseUrl();
        return new BlueprintDraftEntity(contentDraft, baseUrl);
    }

    private void enableFirstBlueprintCreationFlashScope(PageEntity result) {
        // Persist FlashScope variable to display index popup in the next page
        String flashId = FlashScope.persist();
        result.setCreateSuccessRedirectUrl(FlashScope.getFlashScopeUrl(result.getUrl(), flashId));
        if (result.getIndexPage() != null) {
            result.getIndexPage().setCreateSuccessRedirectUrl(FlashScope.getFlashScopeUrl(result.getIndexPage().getUrl(), flashId));
        }
    }
}
