package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.api.service.accessmode.AccessModeService;
import com.atlassian.confluence.api.service.accessmode.ReadOnlyAccessAllowed;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintCleaner;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.SpaceBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.extensions.BlueprintModuleDescriptor;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.Collection;

/**
 * REST methods for administrating Blueprints at a high-level.
 *
 * @since 2.0
 */
@Path("/admin")
public class BlueprintsAdminResource extends AbstractRestResource {
    private final SpaceBlueprintManager spaceBlueprintManager;
    private final ContentBlueprintManager contentBlueprintManager;
    private final ContentTemplateRefManager contentTemplateManager;
    private final PluginAccessor pluginAccessor;
    private final ContentBlueprintCleaner contentBlueprintCleaner;

    public BlueprintsAdminResource(
            final @Qualifier("spaceBlueprintManager") SpaceBlueprintManager spaceBlueprintManager,
            final ContentBlueprintManager contentBlueprintManager,
            final ContentTemplateRefManager contentTemplateManager,
            final @ComponentImport PermissionManager permissionManager,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport PluginAccessor pluginAccessor,
            final @ComponentImport AccessModeService accessModeService,
            final ContentBlueprintCleaner contentBlueprintCleaner) {
        super(permissionManager, spaceManager, accessModeService);

        this.spaceBlueprintManager = spaceBlueprintManager;
        this.contentBlueprintManager = contentBlueprintManager;
        this.contentTemplateManager = contentTemplateManager;
        this.pluginAccessor = pluginAccessor;
        this.contentBlueprintCleaner = contentBlueprintCleaner;
    }

    @POST
    @ReadOnlyAccessAllowed
    @Path("refreshAoTables")
    @Consumes({MediaType.APPLICATION_JSON})
    public int refreshAoTables() {
        checkAdminPermission();

        // Ordering here is important due to non-cascading AO deletes.
        spaceBlueprintManager.deleteAll();
        contentBlueprintManager.deleteAll();
        contentTemplateManager.deleteAll();

        int refreshed = refreshContentBlueprints();
        refreshed += spaceBlueprintManager.getAll().size(); // forces load from plugin system - better than Content BP

        return refreshed;
    }

    @DELETE
    @ReadOnlyAccessAllowed
    @Path("cleanup")
    public Response cleanUp() {
        checkAdminPermission();

        int totalSpaces = contentBlueprintCleaner.cleanUp();
        return totalSpaces > -1 ? Response.ok().entity(totalSpaces).build() : Response.serverError().build();
    }

    // Nasty, but Edu is refactoring the contentBlueprintManager so it should end up with a getAll-like method.
    private int refreshContentBlueprints() {
        final Collection<BlueprintModuleDescriptor> moduleDescriptors = pluginAccessor.getEnabledModuleDescriptorsByClass(BlueprintModuleDescriptor.class);
        for (BlueprintModuleDescriptor moduleDescriptor : moduleDescriptors) {
            contentBlueprintManager.getPluginBackedContentBlueprint(moduleDescriptor.getBlueprintKey(), null);
        }
        return moduleDescriptors.size();
    }
}
