package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.services.TemplateUpdater;
import com.atlassian.plugin.ModuleCompleteKey;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;

@Component
public class DefaultTemplateUpdater implements TemplateUpdater {
    private final ContentBlueprintManager contentBlueprintManager;

    @Autowired
    public DefaultTemplateUpdater(final ContentBlueprintManager contentBlueprintManager) {
        this.contentBlueprintManager = contentBlueprintManager;
    }

    @Override
    public void updateContentTemplateRef(PageTemplate template) {
        updateBlueprintForTemplate(template, false);
    }

    @Override
    public void revertContentTemplateRef(PageTemplate template) {
        updateBlueprintForTemplate(template, true);
    }

    private void updateBlueprintForTemplate(PageTemplate template, boolean revertTemplateRef) {
        // Only edited blueprint templates have referencing keys, skip the rest as they are the original ones
        String blueprintPluginKey = template.getReferencingPluginKey();
        String blueprintModuleKey = template.getReferencingModuleKey();

        if (StringUtils.isBlank(blueprintPluginKey))
            return;                                 // it's not a template defined in a blueprint plugin: carry on your way
        if (StringUtils.isBlank(blueprintModuleKey))
            throw new IllegalStateException("PageTemplate is in invalid state, no module key but plugin key is: " + blueprintPluginKey);

        ModuleCompleteKey blueprintModuleCompleteKey =
                new ModuleCompleteKey(blueprintPluginKey, blueprintModuleKey);

        // Gets the ContentBlueprint customised if existing, otherwise creates one
        ContentBlueprint contentBlueprint = contentBlueprintManager.getOrCreateCustomBlueprint(blueprintModuleCompleteKey, template.getSpace());

        // Gets the content template from the blueprint which reference is to be modified
        ContentTemplateRef contentTemplateRef = findUpdatedContentTemplateRef(template.getModuleCompleteKey(),
                contentBlueprint);

        // The original templateId value for the contentTemplateRef is 0, so if reverting set it back to that.
        long templateId = revertTemplateRef ? 0 : template.getId();
        contentTemplateRef.setTemplateId(templateId);

        // The contentTemplateRef in the Content Blueprint now points to the correct PageTemplate in the DB. Persist!
        contentBlueprintManager.update(contentBlueprint);
    }

    private ContentTemplateRef findUpdatedContentTemplateRef(ModuleCompleteKey moduleCompleteKey,
                                                             ContentBlueprint contentBlueprint) {
        String updatedTemplateModuleKey = moduleCompleteKey.getCompleteKey();
        List<ContentTemplateRef> contentTemplateRefs = contentBlueprint.getContentTemplateRefs();
        for (ContentTemplateRef contentTemplateRef : contentTemplateRefs) {
            String existingKey = contentTemplateRef.getModuleCompleteKey();
            if (existingKey.equals(updatedTemplateModuleKey)) {
                // MATCH! this template referred to by the Content Blueprint is the one that was just updated
                return contentTemplateRef;
            }
        }

        // Might be the blueprint's index page then, if it has one.
        ContentTemplateRef indexPageTemplateRef = contentBlueprint.getIndexPageTemplateRef();
        if (indexPageTemplateRef != null && indexPageTemplateRef.getModuleCompleteKey().equals(updatedTemplateModuleKey)) {
            return indexPageTemplateRef;
        }

        throw new IllegalStateException("No matching content-template ref found for blueprint: " + contentBlueprint.getModuleCompleteKey());
    }
}
