package com.atlassian.confluence.plugins.createcontent.listeners;

import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.plugins.createcontent.TemplatePageCreateEvent;
import com.atlassian.confluence.plugins.createcontent.actions.BlueprintManager;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.List;
import java.util.Map;

import static com.atlassian.confluence.plugins.createcontent.actions.DefaultBlueprintContentGenerator.CONTENT_PAGE_TITLE_CONTEXT_KEY;
import static com.atlassian.confluence.plugins.createcontent.actions.DefaultBlueprintContentGenerator.PAGE_TITLE_PREFIX_CONTEXT_KEY;
import static com.atlassian.confluence.plugins.createcontent.actions.DefaultBlueprintContentGenerator.USE_PAGE_TEMPLATE_TITLE_CONTEXT_KEY;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.NO_PAGE_TITLE_PREFIX;

/**
 * Creates child Templates of a page after it is created.
 *
 * @since 1.6
 */
@Component
public class TemplateChildCreatorListener {
    private final BlueprintManager blueprintManager;
    private final EventPublisher eventPublisher;

    @Autowired
    public TemplateChildCreatorListener(
            final BlueprintManager blueprintManager,
            final @ComponentImport EventPublisher eventPublisher) {
        this.blueprintManager = blueprintManager;
        this.eventPublisher = eventPublisher;
    }

    @EventListener
    public void onCreateEvent(final TemplatePageCreateEvent event) {
        ContentTemplateRef templateRef = event.getTemplateRef();
        List<ContentTemplateRef> children = templateRef.getChildren();
        if (children.isEmpty())
            return;

        Page parentPage = event.getPage();
        String parentTitle = parentPage.getTitle();

        Map<String, Object> originalContext = event.getContext();

        boolean addPageTitlePrefix = !"true".equals(originalContext.get(NO_PAGE_TITLE_PREFIX.key()));

        for (ContentTemplateRef child : children) {
            Map<String, Object> childContext = Maps.newHashMap(originalContext);

            // CONTENT_PAGE_TITLE_CONTEXT_KEY will be the parent-page key; unset it.
            childContext.remove(CONTENT_PAGE_TITLE_CONTEXT_KEY);
            childContext.put(USE_PAGE_TEMPLATE_TITLE_CONTEXT_KEY, true);
            if (addPageTitlePrefix) {
                childContext.put(PAGE_TITLE_PREFIX_CONTEXT_KEY, parentTitle);
            }

            // TODO - expose method in this manager to create page from template.
            blueprintManager.createPageFromTemplate(child, event.getUser(), parentPage.getSpace(),
                    parentPage, childContext, event.getSaveContext());
        }
    }

    @PreDestroy
    public void destroy() {
        eventPublisher.unregister(this);
    }

    @PostConstruct
    public void afterPropertiesSet() {
        eventPublisher.register(this);
    }
}
