package com.atlassian.confluence.plugins.createcontent.listeners;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.confluence.plugins.createcontent.AoBackedManager;
import com.atlassian.confluence.plugins.createcontent.BlueprintConstants;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.SpaceBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.exceptions.BlueprintPluginNotFoundException;
import com.atlassian.confluence.plugins.createcontent.extensions.BlueprintModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.extensions.ContentTemplateModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.extensions.SpaceBlueprintModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.plugins.createcontent.impl.PluginBackedBlueprint;
import com.atlassian.confluence.plugins.createcontent.impl.SpaceBlueprint;
import com.atlassian.event.api.AsynchronousPreferred;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.event.events.PluginEnabledEvent;
import com.atlassian.plugin.event.events.PluginFrameworkStartedEvent;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;

@Component
public class PluginEnabledListener {
    private static final Logger log = LoggerFactory.getLogger(PluginEnabledListener.class);

    private static final String KEY_CC_PLUGIN = "com.atlassian.confluence.plugins.confluence-create-content-plugin";
    private final ContentBlueprintManager contentBlueprintManager;
    private final SpaceBlueprintManager spaceBlueprintManager;
    private final ContentTemplateRefManager contentTemplateRefManager;
    private final PluginAccessor pluginAccessor;
    private final EventPublisher eventPublisher;
    private final TransactionTemplate transactionTemplate;
    private final ActiveObjects activeObjects;
    final AtomicBoolean initialPluginsScanned = new AtomicBoolean(false);

    @Autowired
    public PluginEnabledListener(
            final ContentBlueprintManager contentBlueprintManager,
            final SpaceBlueprintManager spaceBlueprintManager,
            final ContentTemplateRefManager contentTemplateRefManager,
            final @ComponentImport PluginAccessor pluginAccessor,
            final @ComponentImport EventPublisher eventPublisher,
            final @ComponentImport TransactionTemplate transactionTemplate,
            final @ComponentImport ActiveObjects activeObjects) {
        this.contentBlueprintManager = contentBlueprintManager;
        this.spaceBlueprintManager = spaceBlueprintManager;
        this.contentTemplateRefManager = contentTemplateRefManager;
        this.pluginAccessor = pluginAccessor;
        this.eventPublisher = eventPublisher;
        this.transactionTemplate = transactionTemplate;
        this.activeObjects = activeObjects;
    }

    @PostConstruct
    public void postConstruct() {
        eventPublisher.register(this);
    }

    @PreDestroy
    public void preDestroy() {
        eventPublisher.unregister(this);
    }

    @EventListener
    public void onPluginEnabledEvent(PluginEnabledEvent event) {
        if (!initialPluginsScanned.get() || KEY_CC_PLUGIN.equals(event.getPlugin().getKey())) {
            return;
        }
        eventPublisher.publish(new AsyncPluginEnabledEvent(event.getPlugin()));
    }

    @EventListener
    public void onAsyncPluginEnabledEvent(final AsyncPluginEnabledEvent event) {
        activeObjects.flushAll();
        updatePluginModuleMetadata(event.getPlugin());
    }

    @EventListener
    public void onPluginFrameworkStartedEvent(final PluginFrameworkStartedEvent event) {
        initialPluginsScanned.getAndSet(true);
        log.debug("Plugins have finished loading. Flushing all Active Objects tables before scanning for blueprints.");
        activeObjects.flushAll();
        Collection<Plugin> enabledPlugins = pluginAccessor.getEnabledPlugins();
        enabledPlugins.removeIf(plugin -> KEY_CC_PLUGIN.equals(plugin.getKey()));
        enabledPlugins.forEach(this::updatePluginModuleMetadata);
    }

    void updatePluginModuleMetadata(final Plugin plugin) {
        transactionTemplate.execute((TransactionCallback<Void>) () -> {
            Collection<ModuleDescriptor<?>> moduleDescriptors = plugin.getModuleDescriptors();
            for (ModuleDescriptor<?> module : moduleDescriptors) {
                if (module instanceof BlueprintModuleDescriptor) {
                    parse((BlueprintModuleDescriptor) module);
                } else if (module instanceof SpaceBlueprintModuleDescriptor) {
                    parse((SpaceBlueprintModuleDescriptor) module);
                }
            }
            return null;
        });
    }

    private void parse(final BlueprintModuleDescriptor module) {
        ContentBlueprint storedPluginClone = contentBlueprintManager.getCloneByModuleCompleteKey(module.getBlueprintKey());
        if (storedPluginClone == null) {
            return;
        }

        // Index template
        ModuleCompleteKey indexTemplateKey = module.getIndexTemplate();
        ContentTemplateRef indexTemplateRef = storedPluginClone.getIndexPageTemplateRef();
        ContentTemplateRef newIndexTemplateRef = deleteCreateUpdate(indexTemplateKey, indexTemplateRef);
        storedPluginClone.setIndexPageTemplateRef(newIndexTemplateRef);

        // Content templates
        List<ModuleCompleteKey> contentTemplateKeys = module.getContentTemplates();
        List<ContentTemplateRef> dbContentTemplates = storedPluginClone.getContentTemplateRefs();
        deleteCreateUpdate(contentTemplateKeys, dbContentTemplates);

        update(module, storedPluginClone);
    }

    private void parse(@Nonnull final SpaceBlueprintModuleDescriptor module) {
        if (module.getCompleteKey().equals(BlueprintConstants.MODULE_KEY_BLANK_SPACE)) {
            return;
        }
        ModuleCompleteKey moduleCompleteKey = new ModuleCompleteKey(module.getCompleteKey());
        SpaceBlueprint spaceBlueprint = spaceBlueprintManager.getCloneByModuleCompleteKey(moduleCompleteKey);
        if (spaceBlueprint == null) {
            return;
        }

        SpaceBlueprintModuleDescriptor.ContentTemplateRefNode contentTemplateRefNode = module.getContentTemplateRefNode();

        UUID homePageId = spaceBlueprint.getHomePageId();
        if (homePageId != null) {
            ContentTemplateRef homePage = contentTemplateRefManager.getById(homePageId);
            if (homePage == null) {
                throw new RuntimeException("ContentTemplate with UUID " + homePageId.toString() + " not found!");
            }

            if (contentTemplateRefNode != null) {
                parseHomepageChildren(contentTemplateRefNode, homePage);
                deleteCreateUpdate(contentTemplateRefNode.ref, homePage);
            } else {
                spaceBlueprint.setHomePageId(null);
            }
        } else {
            if (contentTemplateRefNode != null) {
                final ContentTemplateRef homePageRef = deleteCreateUpdate(contentTemplateRefNode.ref, null);
                spaceBlueprint.setHomePageId(homePageRef.getId());
            }
        }

        update(module, spaceBlueprint);

        if (homePageId != null && spaceBlueprint.getHomePageId() == null) {
            deleteIfExists(homePageId);
        }
    }

    private void parseHomepageChildren(@Nonnull final SpaceBlueprintModuleDescriptor.ContentTemplateRefNode descNode, @Nonnull final ContentTemplateRef homePage) {
        final List<SpaceBlueprintModuleDescriptor.ContentTemplateRefNode> children = descNode.children;
        if (children != null && !children.isEmpty()) {
            final List<ContentTemplateRef> dbContentTemplates = homePage.getChildren();
            final List<ContentTemplateRef> toRemove = new ArrayList<>(dbContentTemplates);

            for (SpaceBlueprintModuleDescriptor.ContentTemplateRefNode child : children) {
                ContentTemplateRef dbObj = findByKey(dbContentTemplates, child.ref);
                boolean needsAdding = false;
                if (dbObj != null) {
                    toRemove.remove(dbObj);
                } else {
                    needsAdding = true;
                }
                dbObj = deleteCreateUpdate(child.ref, dbObj);
                if (dbObj != null) {
                    if (needsAdding)
                        homePage.addChildTemplateRef(dbObj);
                    parseHomepageChildren(child, dbObj);
                }
            }
            delete(toRemove);
            dbContentTemplates.removeAll(toRemove);
        }
    }

    private ContentTemplateRef findByKey(final List<ContentTemplateRef> dbList, final ModuleCompleteKey key) {
        String strKey = key.getCompleteKey();
        for (ContentTemplateRef contentTemplateRef : dbList) {
            if (strKey.equals(contentTemplateRef.getModuleCompleteKey()))
                return contentTemplateRef;

            List<ContentTemplateRef> children = contentTemplateRef.getChildren();
            if (!children.isEmpty()) {
                ContentTemplateRef result = findByKey(children, key);
                if (result != null)
                    return result;
            }
        }

        return null;
    }

    // CRUD methods

    private ContentTemplateRef create(final ContentTemplateModuleDescriptor desc) {
        final ContentTemplateRef dbObj = new ContentTemplateRef(null, 0, desc.getCompleteKey(), i18n(desc), true, null);
        return contentTemplateRefManager.create(dbObj);
    }

    private void update(@Nonnull final BlueprintModuleDescriptor desc, @Nonnull final ContentBlueprint dbObj) {
        dbObj.setCreateResult(desc.getCreateResult());
        dbObj.setHowToUseTemplate(desc.getHowToUseTemplate());
        dbObj.setIndexKey(desc.getIndexKey());
        dbObj.setIndexTitleI18nKey(desc.getIndexTitleI18nKey());
        String i18nNameKey = i18n(desc, false);
        if (i18nNameKey == null)
            i18nNameKey = desc.getName();
        dbObj.setI18nNameKey(i18nNameKey);

        updateNonClones(contentBlueprintManager, dbObj, i18nNameKey);

        contentBlueprintManager.update(dbObj);
    }

    private void update(@Nonnull final ContentTemplateModuleDescriptor desc, @Nonnull final ContentTemplateRef dbObj) {
        String i18nNameKey = i18n(desc);
        dbObj.setI18nNameKey(i18nNameKey);

        updateNonClones(contentTemplateRefManager, dbObj, i18nNameKey);

        contentTemplateRefManager.update(dbObj);
    }

    private void update(@Nonnull final SpaceBlueprintModuleDescriptor desc, @Nonnull final SpaceBlueprint dbObj) {
        String i18nNameKey = i18n(desc);
        dbObj.setI18nNameKey(i18nNameKey);
        dbObj.setPromotedBps(desc.getPromotedBlueprintKeys());
        dbObj.setCategory(desc.getCategory());

        updateNonClones(spaceBlueprintManager, dbObj, i18nNameKey);

        spaceBlueprintManager.update(dbObj);
    }

    private <O extends PluginBackedBlueprint, M extends AoBackedManager<O, ?>> void updateNonClones(final M manager, final O dbObj, final String i18nNameKey) {
        List<O> nonClones = manager.getNonClonesByModuleCompleteKey(new ModuleCompleteKey(dbObj.getModuleCompleteKey()));
        for (O nonClone : nonClones) {
            nonClone.setI18nNameKey(i18nNameKey);
            manager.update(nonClone);
        }
    }

    private void delete(@Nonnull final List<ContentTemplateRef> dbObjs) {
        for (ContentTemplateRef dbObj : dbObjs) {
            // TODO: Maybe add a delete with list?
            contentTemplateRefManager.delete(dbObj.getId());
        }
    }

    @Nullable
    private ContentTemplateRef deleteCreateUpdate(@Nullable final ModuleCompleteKey descKey, @Nullable ContentTemplateRef dbObj) {
        if (descKey == null) {
            deleteIfExists(dbObj);
            dbObj = null;
        } else {
            final String completeKey = descKey.getCompleteKey();
            if (dbObj != null && !completeKey.equals(dbObj.getModuleCompleteKey())) {
                deleteIfExists(dbObj);
                dbObj = null;
            }
            ContentTemplateModuleDescriptor desc = (ContentTemplateModuleDescriptor) pluginAccessor.getPluginModule(completeKey);
            if (desc == null)
                throw new BlueprintPluginNotFoundException("Module with key " + completeKey + " not found!", ResourceErrorType.NOT_FOUND_CONTENT_TEMPLATE_REF, completeKey);

            if (dbObj == null)
                dbObj = create(desc);
            else
                update(desc, dbObj);

        }
        return dbObj;
    }

    private void deleteCreateUpdate(@Nonnull final List<ModuleCompleteKey> descKeys, @Nonnull final List<ContentTemplateRef> dbObjs) {
        final List<ContentTemplateRef> toRemove = new ArrayList<>(dbObjs);

        for (ModuleCompleteKey descKey : descKeys) {
            final ContentTemplateRef dbObj = findByKey(dbObjs, descKey);

            // If the module is not found, this will thrown an exception here
            final ContentTemplateModuleDescriptor desc = (ContentTemplateModuleDescriptor) pluginAccessor.getPluginModule(descKey.getCompleteKey());

            if (dbObj == null) {
                ContentTemplateRef newDbObj = create(desc);
                dbObjs.add(newDbObj);
            } else {
                // Really, we're saving unnecessarily, because updating the blueprint will resave it for us. However, we need to update the values on the object
                // This method is just called when updating a plugin, so not a really big deal
                update(desc, dbObj);
                toRemove.remove(dbObj);
            }
        }

        delete(toRemove);
        dbObjs.removeAll(toRemove);
    }

    private void deleteIfExists(@Nullable final ContentTemplateRef dbObj) {
        if (dbObj == null)
            return;

        deleteIfExists(dbObj.getId());
    }

    private void deleteIfExists(@Nonnull final UUID id) {
        contentTemplateRefManager.delete(id);
    }

    @Nonnull
    private String i18n(@Nonnull AbstractModuleDescriptor desc) {
        return i18n(desc, true);
    }

    @Nullable
    private String i18n(@Nonnull AbstractModuleDescriptor desc, boolean failIfNotFound) {
        final String i18nNameKey = desc.getI18nNameKey();
        if (i18nNameKey == null) {
            final String message = "i18n-name-key must be specified for: " + desc.getCompleteKey();
            log.warn(message);

            if (failIfNotFound) {
                throw new IllegalArgumentException(message);
            }
        }

        return i18nNameKey;
    }

    /**
     * This event is required because OSGi is single threaded, and AO times out waiting for AOConfiguration which will
     * never become available if we wait for it on the OSGi thread. Without it being asynchronous, any plugin that has
     * imports on Create Content Plugin will fail to get re-enabled.
     */
    @AsynchronousPreferred
    static class AsyncPluginEnabledEvent {
        private final Plugin plugin;

        public AsyncPluginEnabledEvent(Plugin plugin) {
            this.plugin = plugin;
        }

        public Plugin getPlugin() {
            return plugin;
        }
    }
}
