package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.core.ContentPropertyManager;
import com.atlassian.confluence.plugins.createcontent.api.services.ContentBlueprintSanitiserManager;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateBlueprintPageRestEntity;
import com.atlassian.confluence.plugins.createcontent.services.RequestStorage;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageEntity;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.web.context.HttpContext;
import org.codehaus.jackson.map.ObjectMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpSession;
import java.io.IOException;

import static com.atlassian.confluence.plugins.createcontent.scheduled.CreateBlueprintDraftPageEntityCleanupJob.CLEANED_RECORD_KEY;
import static org.apache.commons.lang3.StringUtils.isBlank;

/*
    This storage originally used a contentPropertyManager and stored against the Draft id, but this doesn't
    work for Drafts by anonymous users (the same draft with id 0 would be used for all anonymous storage).
    Given that all create-request drafts are expected to be converted to pages within the same session we can
    use the Session for storage.

    If we decide to reuse the contentPropertyManager later (CONFDEV-18341), be sure to look at old versions of
    this class!
 */
@Component
public class DefaultRequestStorage implements RequestStorage {
    public static final String DRAFT_CREATE_REQUEST = "create.blueprint.page.draft.request";

    private final HttpContext httpContext;
    private final ContentPropertyManager contentPropertyManager;
    private final ObjectMapper objectMapper;
    private final ContentBlueprintSanitiserManager sanitiserManager;

    @Autowired
    public DefaultRequestStorage(
            final @ComponentImport HttpContext httpContext,
            final @ComponentImport ContentPropertyManager contentPropertyManager,
            final ContentBlueprintSanitiserManager sanitiserManager) {
        this.httpContext = httpContext;
        this.contentPropertyManager = contentPropertyManager;
        this.sanitiserManager = sanitiserManager;
        objectMapper = new ObjectMapper();
    }

    @Override
    public void storeCreateRequest(CreateBlueprintPageEntity createRequest, ContentEntityObject ceo) {
        // CONFSRVDEV-9710: remove PII if it's present using the sanitiser manager
        CreateBlueprintPageEntity sanitisedRequest = sanitiserManager.sanitise(createRequest);
        if (ceo.isPersistent()) {
            String jsonRequest;
            try {
                jsonRequest = objectMapper.writeValueAsString(sanitisedRequest);
                contentPropertyManager.setTextProperty(ceo, DRAFT_CREATE_REQUEST, jsonRequest);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        } else {
            getSession().setAttribute(getAttributeKey(ceo), sanitisedRequest);
        }
    }

    @Override
    public CreateBlueprintPageEntity retrieveRequest(ContentEntityObject ceo) {
        CreateBlueprintPageEntity request;

        if (ceo.isPersistent()) {
            String sanitisedJsonRequest = contentPropertyManager.getTextProperty(ceo, DRAFT_CREATE_REQUEST);
            if (isBlank(sanitisedJsonRequest))
                throw new IllegalStateException("No persisted CreateBlueprint request found for draft with id: " + ceo.getId());

            try {
                request = objectMapper.readValue(sanitisedJsonRequest, CreateBlueprintPageRestEntity.class);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }

        } else {
            request = (CreateBlueprintPageEntity) getSession().getAttribute(getAttributeKey(ceo));
        }

        return sanitiserManager.unsanitise(request);
    }

    // TODO - This method can be deleted once reliable-save has been finalised
    // See https://jira.atlassian.com/browse/CONFDEV/component/37892
    @Override
    public void clear(ContentEntityObject ceo) {
        if (ceo.isPersistent()) {
            contentPropertyManager.removeProperty(ceo, DRAFT_CREATE_REQUEST);
            contentPropertyManager.removeProperty(ceo, CLEANED_RECORD_KEY);
        } else {
            getSession().removeAttribute(getAttributeKey(ceo));
        }
    }

    private String getAttributeKey(ContentEntityObject ceo) {
        return DRAFT_CREATE_REQUEST + ceo.getIdAsString();
    }

    private HttpSession getSession() {
        return httpContext.getSession(true);
    }
}