package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.core.DateFormatter;
import com.atlassian.confluence.core.FormatSettingsManager;
import com.atlassian.confluence.core.TimeZone;
import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.plugins.createcontent.api.services.CreateButtonService;
import com.atlassian.confluence.plugins.createcontent.services.BlueprintResolver;
import com.atlassian.confluence.renderer.template.TemplateRenderer;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.confluence.web.UrlBuilder;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.message.I18nResolver;
import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isBlank;

@Component
@ExportAsService(CreateButtonService.class)
public class DefaultCreateButtonService implements CreateButtonService {
    private static final String SPACE_KEY = "spaceKey";
    private static final String TEMPLATE_ID = "templateId";
    private static final String BUTTON_LABEL = "buttonLabel";
    private static final String TITLE = "title";
    private static final String HAS_CREATE_PERMISSION = "hasCreatePermission";
    private static final String CREATE_CONTENT_URL = "createContentUrl";
    private static final String CONTENT_BLUEPRINT_ID = "contentBlueprintId";
    private static final String BUTTON_LABEL_DEFAULT_KEY =
            "com.atlassian.confluence.plugins.confluence-create-content-plugin.create-from-template.param.buttonLabel.default-value";

    private final UserAccessor userAccessor;
    private final FormatSettingsManager formatSettingsManager;
    private final LocaleManager localeManager;
    private final I18nResolver i18nResolver;
    private final TemplateRenderer templateRenderer;
    private final PermissionManager permissionManager;
    private final SettingsManager settingsManager;
    private final BlueprintResolver blueprintResolver;

    @Autowired
    public DefaultCreateButtonService(
            final @ComponentImport UserAccessor userAccessor,
            final @ComponentImport FormatSettingsManager formatSettingsManager,
            final @ComponentImport LocaleManager localeManager,
            final @ComponentImport I18nResolver i18nResolver,
            final @ComponentImport TemplateRenderer templateRenderer,
            final @ComponentImport PermissionManager permissionManager,
            final @ComponentImport SettingsManager settingsManager,
            final BlueprintResolver blueprintResolver) {
        this.userAccessor = userAccessor;
        this.formatSettingsManager = formatSettingsManager;
        this.localeManager = localeManager;
        this.i18nResolver = i18nResolver;
        this.templateRenderer = templateRenderer;
        this.permissionManager = permissionManager;
        this.settingsManager = settingsManager;
        this.blueprintResolver = blueprintResolver;
    }

    @Override
    public String renderBlueprintButton(Space space, String contentBlueprintId, String blueprintModuleCompleteKey, String buttonLabelKey, String newPageTitle) {
        return renderButton(space, contentBlueprintId, blueprintModuleCompleteKey, 0, buttonLabelKey, newPageTitle);
    }

    @Override
    public String renderTemplateButton(Space space, long templateId, String buttonLabelKey, String newPageTitle) {
        return renderButton(space, null, null, templateId, buttonLabelKey, newPageTitle);
    }

    private String renderButton(Space space, String contentBlueprintId, String blueprintModuleCompleteKey, long templateId, String buttonLabelKey, String pageTitle) {
        ContentBlueprint contentBlueprint = null;

        Map<String, Object> context = Maps.newHashMap();
        if (templateId != 0) {
            context.put(TEMPLATE_ID, templateId);
        } else {
            contentBlueprint = blueprintResolver.getContentBlueprint(contentBlueprintId, blueprintModuleCompleteKey, space.getKey());
            context.put(CONTENT_BLUEPRINT_ID, contentBlueprint.getId().toString());
        }

        // Always try to resolve the i18n key of the label - if it's already an i18n'd string getText will return it as is.
        if (isBlank(buttonLabelKey))
            buttonLabelKey = BUTTON_LABEL_DEFAULT_KEY;
        String buttonLabel = i18nResolver.getText(buttonLabelKey);
        context.put(BUTTON_LABEL, buttonLabel);

        String title = getTitle(pageTitle, space);
        if (title != null)
            context.put(TITLE, title);

        String spaceKey = space.getKey();
        context.put(SPACE_KEY, spaceKey);

        context.put(HAS_CREATE_PERMISSION, permissionManager.hasCreatePermission(getUser(), space, Page.class));
        context.put(CREATE_CONTENT_URL, getCreateContentUrl(contentBlueprint, templateId, space, title));

        StringBuilder output = new StringBuilder();
        templateRenderer.renderTo(output, "com.atlassian.confluence.plugins.confluence-create-content-plugin:create-from-template-resources", "Confluence.Templates.Blueprints.CreateFromTemplate.macroTemplate.soy", context);

        return output.toString();
    }

    private ConfluenceUser getUser() {
        return AuthenticatedUserThreadLocal.get();
    }

    private String getTitle(String title, Space space) {
        if (isBlank(title))
            return title;

        title = title.replaceAll("(?<!\\\\)@spaceName", space.getName());
        title = title.replaceAll("(?<!\\\\)@spaceKey", space.getKey());

        TimeZone timeZone = userAccessor.getConfluenceUserPreferences(AuthenticatedUserThreadLocal.get()).getTimeZone();
        title = title.replaceAll("(?<!\\\\)@currentDate",
                new DateFormatter(timeZone, formatSettingsManager, localeManager).getDateForBlogPost(new Date()));

        return title;
    }

    /**
     * @return a URL for emails that may include a rendered macro button.
     */
    private String getCreateContentUrl(ContentBlueprint blueprint, long templateId, Space space, String title) {
        final String baseUrl = settingsManager.getGlobalSettings().getBaseUrl();
        UrlBuilder createContentUrl = new UrlBuilder(baseUrl);

        String spaceKey = space.getKey();
        if (templateId != 0) {
            createContentUrl.add("templateId", templateId);
            createContentUrl.add("spaceKey", spaceKey);
            createContentUrl.add("newSpaceKey", spaceKey);
        } else { // blueprint
            createContentUrl.add("createDialogSpaceKey", spaceKey);
            createContentUrl.add("createDialogBlueprintId", blueprint.getId().toString());
        }

        if (title != null) {
            createContentUrl.add("title", title);
        }

        return createContentUrl.toString();
    }
}
