package com.atlassian.confluence.plugins.createcontent.extensions;

import com.atlassian.confluence.plugins.dialog.wizard.api.DialogWizard;
import com.atlassian.confluence.plugins.dialog.wizard.api.DialogWizardModuleDescriptor;
import com.atlassian.confluence.util.i18n.DocumentationBeanFactory;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.module.ModuleFactory;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * Represents a Space blueprint that creates Confluence Spaces in a pluggable fashion.
 *
 * @since 1.6
 */
public class SpaceBlueprintModuleDescriptor extends AbstractModuleDescriptor<Void> {
    public static class ContentTemplateRefNode {
        public ModuleCompleteKey ref;
        public List<ContentTemplateRefNode> children;

        public ContentTemplateRefNode(ModuleCompleteKey ref) {
            this.ref = ref;
        }
    }

    private static final Logger log = LoggerFactory.getLogger(SpaceBlueprintModuleDescriptor.class);

    private final DocumentationBeanFactory documentationBeanFactory;

    private ContentTemplateRefNode contentTemplateRefNode;

    private Element tempWizardElement;

    private DialogWizard dialogWizard;

    private List<ModuleCompleteKey> promotedBlueprintKeys = newArrayList();

    private String category;

    public SpaceBlueprintModuleDescriptor(
            @ComponentImport ModuleFactory moduleFactory,
            @ComponentImport DocumentationBeanFactory documentationBeanFactory) {
        super(moduleFactory);

        this.documentationBeanFactory = documentationBeanFactory;
    }

    @Override
    public void init(@Nonnull Plugin plugin, @Nonnull Element element) throws PluginParseException {
        super.init(plugin, element);

        if (isBlank(getKey()))
            throw new PluginParseException("key is a required attribute of <space-blueprint>.");
        if (isBlank(getI18nNameKey())) {
            // CONFDEV-18602 Should throw exception once our Blueprint plugins are updated
            log.warn("i18n-name-key is a required attribute of <space-blueprint> for module: " + getCompleteKey());
//            throw new PluginParseException("i18n-name-key is a required attribute of <space-blueprint>.");
        }

        Element homePageElement = element.element("content-template");
        if (homePageElement != null) {
            contentTemplateRefNode = makeNode(homePageElement);
        }

        tempWizardElement = element.element("dialog-wizard");

        category = element.attributeValue("category");

        Element promotedBlueprint = element.element("promoted-blueprints");
        if (promotedBlueprint != null) {
            List<Element> blueprintElements = promotedBlueprint.elements("blueprint");
            for (Element keyElement : blueprintElements) {
                promotedBlueprintKeys.add(new ModuleCompleteKey(keyElement.attributeValue("ref")));
            }
        }
    }

    private ContentTemplateRefNode makeNode(Element element) {
        ContentTemplateRefNode result = new ContentTemplateRefNode(new ModuleCompleteKey(plugin.getKey(),
                element.attributeValue("ref")));
        List<Element> children = element.elements("content-template");
        if (children != null) {
            result.children = newArrayList();
            for (Element child : children) {
                // FIXME - could be circular ref in these children
                result.children.add(makeNode(child));
            }
        }

        return result;
    }

    @Override
    public Void getModule() {
        return null;
    }

    public ContentTemplateRefNode getContentTemplateRefNode() {
        return contentTemplateRefNode;
    }

    public DialogWizard getDialogWizard() {
        // important that we calculate the dialog wizard and pass in the documentation bean factory after the blueprint plugin has initialised
        if (dialogWizard == null && tempWizardElement != null) {
            DialogWizardModuleDescriptor moduleDescriptor = new DialogWizardModuleDescriptor(moduleFactory, documentationBeanFactory);
            moduleDescriptor.init(plugin, tempWizardElement);
            dialogWizard = moduleDescriptor.getModule();
            tempWizardElement = null;
        }
        return dialogWizard;
    }

    public List<ModuleCompleteKey> getPromotedBlueprintKeys() {
        return promotedBlueprintKeys;
    }

    public String getCategory() {
        return category;
    }

}
