package com.atlassian.confluence.plugin.module.xmlrpc.mail;

import com.atlassian.confluence.rpc.RemoteException;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.mail.MailException;
import com.atlassian.mail.server.MailServerManager;
import com.atlassian.mail.server.SMTPMailServer;
import com.atlassian.mail.server.impl.SMTPMailServerImpl;
import com.atlassian.user.User;
import org.apache.commons.lang.StringUtils;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public class MailServerHelperServiceDelegate implements MailServerHelperService {

    private MailServerManager mailServerManager;

    private PermissionManager permissionManager;

    public MailServerManager getMailServerManager() {
        return mailServerManager;
    }

    public void setMailServerManager(MailServerManager mailServerManager) {
        this.mailServerManager = mailServerManager;
    }

    public PermissionManager getPermissionManager() {
        return permissionManager;
    }

    public void setPermissionManager(PermissionManager permissionManager) {
        this.permissionManager = permissionManager;
    }

    private boolean isCurrentUserAnAdmin() {
        final User user = AuthenticatedUserThreadLocal.getUser();

        return null != user && getPermissionManager().hasPermission(
                user,
                Permission.ADMINISTER,
                PermissionManager.TARGET_APPLICATION);
    }

    private SMTPMailServer toSMTPMailServer(final Map mailServerStructure) throws RemoteException {
        final String[] smtpServerAddressParts;
        final SMTPMailServer smtpMailServer = new SMTPMailServerImpl(
                new Long((String) mailServerStructure.get("id")),
                (String) mailServerStructure.get("name"),
                null,
                (String) mailServerStructure.get("fromAddress"),
                (String) mailServerStructure.get("prefix"),
                StringUtils.isNotBlank((String) mailServerStructure.get("jndiLocation")),
                (String) mailServerStructure.get("jndiLocation"),
                (String) mailServerStructure.get("userName"),
                (String) mailServerStructure.get("password")
        );

        smtpServerAddressParts = StringUtils.split((String) mailServerStructure.get("address"), ':');
        if (smtpServerAddressParts.length == 1) {
            smtpMailServer.setHostname(smtpServerAddressParts[0]);
            smtpMailServer.setSmtpPort(SMTPMailServer.DEFAULT_SMTP_PORT);
        } else if (smtpServerAddressParts.length == 2) {
            smtpMailServer.setHostname(smtpServerAddressParts[0]);
            smtpMailServer.setSmtpPort(smtpServerAddressParts[1]);
        } else {
            throw new RemoteException("Invalid address specified: " + mailServerStructure.get("address"));
        }

        return smtpMailServer;
    }

    private Hashtable toHashtable(final SMTPMailServer smtpMailServer) throws RemoteException {
        final Hashtable hashtable = new Hashtable();

        if (null != smtpMailServer.getId())
            hashtable.put("id", String.valueOf(smtpMailServer.getId()));
        if (null != smtpMailServer.getName())
            hashtable.put("name", smtpMailServer.getName());
        if (null != smtpMailServer.getDefaultFrom())
            hashtable.put("fromAddress", smtpMailServer.getDefaultFrom());
        if (null != smtpMailServer.getPrefix())
            hashtable.put("prefix", smtpMailServer.getPrefix());
        if (null != smtpMailServer.getJndiLocation())
            hashtable.put("jndiLocation", smtpMailServer.getJndiLocation());
        if (null != smtpMailServer.getUsername())
            hashtable.put("userName", smtpMailServer.getUsername());
        if (null != smtpMailServer.getPassword())
            hashtable.put("password", smtpMailServer.getPassword());
        if (null != smtpMailServer.getHostname() && null != smtpMailServer.getSmtpPort())
            hashtable.put("address", smtpMailServer.getHostname() + ":" + smtpMailServer.getSmtpPort());

        return hashtable;
    }

    private void validateMailServer(SMTPMailServer smtpMailServer) throws RemoteException {
        if (StringUtils.isBlank(smtpMailServer.getName()))
            throw new RemoteException("Mail server name not specified.");
        if (StringUtils.isBlank(smtpMailServer.getDefaultFrom()))
            throw new RemoteException("From address not specified.");
        if (StringUtils.isBlank(smtpMailServer.getPrefix()))
            throw new RemoteException("Prefix not specified.");
        if (StringUtils.isBlank(smtpMailServer.getHostname()) && StringUtils.isBlank(smtpMailServer.getJndiLocation()))
            throw new RemoteException("Server address/JNDI location not specified.");
    }

    public String createMailServer(final String authToken, final Hashtable mailServerStructure) throws RemoteException {
        final SMTPMailServer smtpMailServer;

        if (!isCurrentUserAnAdmin())
            throw new RemoteException("You do not have admin permission.");

        smtpMailServer = toSMTPMailServer(mailServerStructure);
        if (null != smtpMailServer.getId() && smtpMailServer.getId().longValue() != 0)
            throw new RemoteException("Unable to create SMTP server. It seems that there is an ID associated the specified structure: " + smtpMailServer.getId());
        
        validateMailServer(smtpMailServer);

        try {
            return String.valueOf(getMailServerManager().create(smtpMailServer));
        } catch (final MailException me) {
            throw new RemoteException(me);
        }
    }

    public boolean updateMailServer(final String authToken, final Hashtable mailServerStructure) throws RemoteException {
        final SMTPMailServer smtpMailServer;

        if (!isCurrentUserAnAdmin())
            throw new RemoteException("You do not have admin permission.");

        smtpMailServer = toSMTPMailServer(mailServerStructure);
        if (null == smtpMailServer.getId())
            throw new RemoteException("Unable to update SMTP server. There seems to be no ID associated the specified structure");

        validateMailServer(smtpMailServer);

        try {
            getMailServerManager().update(smtpMailServer);
            return true;
        } catch (final MailException me) {
            throw new RemoteException(me);
        }
    }

    public Hashtable readMailServer(final String authToken, final String mailServerId) throws RemoteException {
        if (!isCurrentUserAnAdmin())
            throw new RemoteException("You do not have admin permission.");

        try {
            final List smtpMailServers = getMailServerManager().getSmtpMailServers();

            if (null != smtpMailServers) {
                for (final Iterator i = smtpMailServers.iterator(); i.hasNext();) {
                    final SMTPMailServer smtpMailServer = (SMTPMailServer) i.next();

                    if (StringUtils.equals(String.valueOf(smtpMailServer.getId()), mailServerId))
                        return toHashtable(smtpMailServer);
                }
            }


            return null;

        } catch (final MailException me) {
            throw new RuntimeException(me);
        }
    }

    public boolean deleteMailServer(final String authToken, final String mailServerId) throws RemoteException {
        if (!isCurrentUserAnAdmin())
            throw new RemoteException("You do not have admin permission.");
        
        try {
            getMailServerManager().delete(new Long(mailServerId));
            return true;
        } catch (final MailException me) {
            throw new RuntimeException(me);
        }
    }

    public Hashtable getMailServerIdsAndNames(final String authToken) throws RemoteException {
        if (!isCurrentUserAnAdmin())
            throw new RemoteException("You do not have admin permission.");

        try {
            final List smtpMailServers = getMailServerManager().getSmtpMailServers();
            final Hashtable mailServerIdsAndNames = new Hashtable();

            if (null != smtpMailServers) {
                for (final Iterator i = smtpMailServers.iterator(); i.hasNext();) {
                    final SMTPMailServer smtpMailServer = (SMTPMailServer) i.next();

                    mailServerIdsAndNames.put(String.valueOf(smtpMailServer.getId()), smtpMailServer.getName());
                }
            }

            return mailServerIdsAndNames;

        } catch (final MailException me) {
            throw new RuntimeException(me);
        }
    }

    public String login(final String s, final String s1) throws RemoteException {
        return null;
    }

    public boolean logout(final String s) throws RemoteException {
        return false;
    }
}
