package com.atlassian.confluence.plugin;

import com.atlassian.confluence.ConfluenceSoapService;
import com.atlassian.confluence.ConfluenceSoapServiceServiceLocator;
import com.atlassian.confluence.plugin.helper.*;
import com.meterware.httpunit.UploadFileSpec;
import com.meterware.httpunit.WebForm;
import net.sourceforge.jwebunit.WebTester;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.xmlrpc.XmlRpcException;
import org.apache.xmlrpc.XmlRpcClient;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import javax.xml.rpc.ServiceException;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.Arrays;
import java.util.Vector;


public class ConfluenceWebTester extends WebTester {

    private static final long MAX_REINDEXING_DURATION = 900000; /* 15 minutes */

    private static final Logger LOGGER = Logger.getLogger(ConfluenceWebTester.class);

    private String adminUserName;

    private String adminPassword;

    private String protocol;

    private String hostName;

    private int port;

    private String contextPath;

    private String soapServicePath;

    private String currentUserName;

    private String currentPassword;

    private File confluencePluginJar;

    private String confluencePluginName;

    private File confluenceBackupZip;

    private String licenseString;

    public String getAdminUserName() {
        return adminUserName;
    }

    public void setAdminUserName(String adminUserName) {
        this.adminUserName = adminUserName;
    }

    public String getAdminPassword() {
        return adminPassword;
    }

    public void setAdminPassword(String adminPassword) {
        this.adminPassword = adminPassword;
    }

    public String getProtocol() {
        return protocol;
    }

    public void setProtocol(String protocol) {
        this.protocol = protocol;
    }

    public String getHostName() {
        return hostName;
    }

    public void setHostName(String hostName) {
        this.hostName = hostName;
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public String getContextPath() {
        return contextPath;
    }

    public void setContextPath(String contextPath) {
        this.contextPath = contextPath;
    }

    public String getSoapServicePath() {
        return soapServicePath;
    }

    public void setSoapServicePath(String soapServicePath) {
        this.soapServicePath = soapServicePath;
    }

    public String getCurrentUserName() {
        return currentUserName;
    }

    public void setCurrentUserName(String currentUserName) {
        this.currentUserName = currentUserName;
    }

    public String getCurrentPassword() {
        return currentPassword;
    }

    public void setCurrentPassword(String currentPassword) {
        this.currentPassword = currentPassword;
    }

    public File getConfluencePluginJar() {
        return confluencePluginJar;
    }

    public void setConfluencePluginJar(File confluencePluginJar) {
        this.confluencePluginJar = confluencePluginJar;
    }

    public String getConfluencePluginName() {
        return confluencePluginName;
    }

    public void setConfluencePluginName(String confluencePluginName) {
        this.confluencePluginName = confluencePluginName;
    }

    public File getConfluenceBackupZip() {
        return confluenceBackupZip;
    }

    public void setConfluenceBackupZip(File confluenceBackupZip) {
        this.confluenceBackupZip = confluenceBackupZip;
    }

    public String getLicenseString() {
        return licenseString;
    }

    public void setLicenseString(String licenseString) {
        this.licenseString = licenseString;
    }

    public String getBaseUrl() {
        return new StringBuffer(getProtocol())
                .append("://")
                .append(getHostName())
                .append(':')
                .append(getPort())
                .append(getContextPath())
                .toString();
    }

    public String getSoapServiceLocation() {
        return new StringBuffer(getBaseUrl()).append(getSoapServicePath()).toString();
    }

    public String getXmlRpcServiceLocation() {
        return new StringBuffer(getBaseUrl()).append("/rpc/xmlrpc").toString();
    }

    public void login(final String username, final String password) {
        setCurrentUserName(username);
        setCurrentPassword(password);

        login();
    }

    public void login() {
        gotoPage("/login.action");
        assertTextNotPresent("You are currently logged in");

        setWorkingForm("loginform");
        setFormElement("os_username", getCurrentUserName());
        setFormElement("os_password", getCurrentPassword());
        submit("login");

        assertLinkPresentWithText("Log Out");
    }

    public void logout() {
        gotoPage("/logout.action");
        assertTextPresent("You have been successfully logged out and any automatic logins removed.");
    }

    public XmlRpcClient getXmlRpcClient() throws MalformedURLException {
        return new XmlRpcClient(getXmlRpcServiceLocation());
    }

    public String loginToXmlRpcService(final String userName, final String password) throws XmlRpcException, IOException {
        final XmlRpcClient xmlRpcClient = getXmlRpcClient();

        return (String) xmlRpcClient.execute("confluence1.login",
                new Vector(
                        Arrays.asList(new String[] { userName, password })
                ));
    }

    public void logoutFromXmlRpcService(final String authenticationToken) {
        final XmlRpcClient xmlRpcClient;

        if (StringUtils.isNotBlank(authenticationToken)) {
            try {
                xmlRpcClient = getXmlRpcClient();
                xmlRpcClient.execute("confluence1.logout",
                        new Vector(
                                Arrays.asList(
                                        new String[] { authenticationToken }
                                )
                        ));
            } catch (final XmlRpcException xre) {
                handleInvalidXmlRpcServiceException(xre);
            } catch (final IOException ioe) {
                handleIOException(ioe);
            }
        }
    }

    public String loginToXmlRPcService() throws XmlRpcException, IOException {
        return loginToXmlRpcService(getCurrentUserName(), getCurrentPassword());
    }

    public ConfluenceSoapService getConfluenceSoapService() throws MalformedURLException, ServiceException {
        final ConfluenceSoapServiceServiceLocator confluenceSoapServiceServiceLocator =
                new ConfluenceSoapServiceServiceLocator();

        return confluenceSoapServiceServiceLocator.getConfluenceserviceV1(new URL(getSoapServiceLocation()));
    }

    public String loginToSoapService(final String userName, final String password)
             throws MalformedURLException, ServiceException , RemoteException {
        final ConfluenceSoapService confluenceSoapService = getConfluenceSoapService();

        return StringUtils.isNotBlank(userName) ? confluenceSoapService.login(userName, password) : null;
    }

    public String loginToSoapService() throws MalformedURLException, ServiceException , RemoteException {
        return loginToSoapService(getCurrentUserName(), getCurrentPassword());
    }

    public void logoutFromSoapService(final String authenticationToken) {
        try {
            if (null != authenticationToken)
                getConfluenceSoapService().logout(authenticationToken);
            
        } catch (final MalformedURLException mUrlE) {
            handleInvalidSoapServiceEndpointException(mUrlE);
        } catch (final ServiceException se) {
            handleInvalidSoapServiceException(se);
        } catch (final RemoteException re) {
            handleRemoteException(re);
        }
    }

    protected void updateLicense() throws IOException {
        final String licenseString = getLicenseString();

        if (StringUtils.isNotBlank(licenseString)) {
            gotoPage("/admin/console.action");
            clickLinkWithText("License Details");

            setWorkingForm("updateLicenseForm");
            setFormElement("licenseString", licenseString);

            submit("update");
            assertTextNotPresent("License string is too short");
            assertTextNotPresent("License was not valid");
        }
    }

    protected void refreshLicense() {
        gotoPage("/admin/refreshlicensing.action");
    }

    protected void installPlugin() {
        final File confluencePluginJar = getConfluencePluginJar();

        if (null != confluencePluginJar && confluencePluginJar.isFile()) {
            gotoPage("/admin/plugins.action");
            assertTextPresent("Manage Plugins");

            setWorkingForm("plugin-upload");

            getDialog().getForm().setParameter("file_0", new UploadFileSpec[] { new UploadFileSpec(getConfluencePluginJar()) });
            submit("confirm");

            assertLinkPresentWithText(getConfluencePluginName());
        }
    }

    protected void restoreData() {
        final File confluenceBackupZip = getConfluenceBackupZip();

        if (null != confluenceBackupZip && confluenceBackupZip.isFile()) {
            try {
                final WebForm[] forms;
                final WebForm restoreForm;

                gotoPage("/admin/backup.action?synchronous=true");
                assertTextPresent("Restore Confluence Data");

                forms = getDialog().getResponse().getForms();
                restoreForm = forms[2];

                restoreForm.setParameter("file", new UploadFileSpec[] { new UploadFileSpec( confluenceBackupZip ) });
                restoreForm.submit();
                
                reindex();

            } catch (final SAXException saxE) {
                throw new RuntimeException(saxE);
            } catch (final IOException ioe) {
                throw new RuntimeException(ioe);
            }
        }
    }
    
    
	public void reindex() {
        final long startIndexingTime = System.currentTimeMillis();
		
		// fire off the reindex action
		gotoPage("/admin/reindex.action");
			
		// Make sure the loop does exit at some point
		while((System.currentTimeMillis() - startIndexingTime) < MAX_REINDEXING_DURATION) {
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {}
			
			// Get the status and look for 100% complete
			gotoPage("/admin/indexsummary.action");
			Element pctComplete = getDialog().getElement("percentComplete");
			if( pctComplete != null ) {
				if(Integer.parseInt(pctComplete.getTextContent()) == 100)
					break;
			}
		}
	}
		    

    public void flushCaches() {
        gotoPage("/admin/console.action");
        clickLinkWithText("Cache Statistics");
        clickLinkWithText("Flush all");
    }
    
    protected PageHelper getPageHelper(final long pageId) {
        final PageHelper pageHelper = new PageHelper();

        pageHelper.setId(pageId);
        pageHelper.setConfluenceWebTester(this);
        
        return pageHelper;
    }

    protected PageHelper getPageHelper() {
        return getPageHelper(0);
    }

    protected BlogPostHelper getBlogPostHelper(final long blogPostId) {
        final BlogPostHelper blogPostHelper = new BlogPostHelper(blogPostId);

        blogPostHelper.setId(blogPostId);
        blogPostHelper.setConfluenceWebTester(this);
        
        return blogPostHelper;
    }

    protected BlogPostHelper getBlogPostHelper() {
        return getBlogPostHelper(0);
    }

    protected SpaceHelper getSpaceHelper(final String spaceKey) {
        final SpaceHelper spaceHelper = new SpaceHelper(spaceKey);

        spaceHelper.setKey(spaceKey);
        spaceHelper.setConfluenceWebTester(this);
        
        return spaceHelper;
    }

    protected SpaceHelper getSpaceHelper() {
        return getSpaceHelper(null);
    }

    protected CommentHelper getCommentHelper(final long commentId) {
        final CommentHelper commentHelper = new CommentHelper(commentId);

        commentHelper.setId(commentId);
        commentHelper.setConfluenceWebTester(this);

        return commentHelper;
    }

    protected CommentHelper getCommentHelper() {
        return getCommentHelper(0);
    }

    protected UserHelper getUserHelper(final String userName) {
        final UserHelper userHelper = new UserHelper(userName);

        userHelper.setName(userName);
        userHelper.setConfluenceWebTester(this);

        return userHelper;
    }

    protected UserHelper getUserHelper() {
        return getUserHelper(null);
    }

    protected GroupHelper getGroupHelper(final String groupName) {
        final GroupHelper groupHelper = new GroupHelper(groupName);

        groupHelper.setName(groupName);
        groupHelper.setConfluenceWebTester(this);

        return groupHelper;
    }

    protected GroupHelper getGroupHelper() {
        return getGroupHelper(null);
    }

    protected MailServerHelper getMailServerHelper(final long id) {
        final MailServerHelper mailServerHelper = new MailServerHelper(id);

        mailServerHelper.setConfluenceWebTester(this);

        return mailServerHelper;
    }

    protected MailServerHelper getMailServerHelper() {
        return getMailServerHelper(0);
    }

    protected AttachmentHelper getAttachmentHelper(final long id, final String fileName) {
        final AttachmentHelper attachmentHelper = new AttachmentHelper(id, fileName);

        attachmentHelper.setConfluenceWebTester(this);

        return attachmentHelper;
    }

    protected AttachmentHelper getAttachmentHelper() {
        return getAttachmentHelper(0, null);
    }

    protected BandanaHelper getBandanaHelper(final String spaceKey, final String bandanaKey) {
        final BandanaHelper bandanaHelper = new BandanaHelper(spaceKey, bandanaKey);

        bandanaHelper.setConfluenceWebTester(this);

        return bandanaHelper;
    }

    protected BandanaHelper getBandanaHelper(final String bandanaKey) {
        return getBandanaHelper(StringUtils.EMPTY, bandanaKey);
    }

    protected BandanaHelper getBandanaHelper() {
        return getBandanaHelper(StringUtils.EMPTY, null);
    }

    public void handleInvalidSoapServiceEndpointException(final MalformedURLException mUrlE) {
        if (LOGGER.isEnabledFor(Level.ERROR))
            LOGGER.error("Invalid Confluence SOAP service endpoint configured: " + getSoapServiceLocation(), mUrlE);
    }

    public void handleInvalidXmlRpcServiceEndpointException(final MalformedURLException mUrlE) {
        if (LOGGER.isEnabledFor(Level.ERROR))
            LOGGER.error("Invalid Confluence XMLRPC service endpoint configured: " + getSoapServiceLocation(), mUrlE);
    }

    public void handleInvalidSoapServiceException(final ServiceException se) {
        if (LOGGER.isEnabledFor(Level.ERROR))
            LOGGER.error("Confluence SOAP service borked.", se);
    }

    public void handleInvalidXmlRpcServiceException(final XmlRpcException e) {
        if (LOGGER.isEnabledFor(Level.ERROR))
            LOGGER.error("Confluence XMLRPC service borked.", e);
    }

    public void handleRemoteException(final RemoteException re) {
        if (LOGGER.isEnabledFor(Level.ERROR))
            LOGGER.error("Requested operation could not be performed.", re);
    }

    public void handleIOException(final IOException e) {
        if (LOGGER.isEnabledFor(Level.ERROR))
            LOGGER.error("Unable to read/write.", e);
    }
}
