package com.atlassian.confluence.ext.usage.event;

import com.atlassian.bonnie.search.Extractor;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.event.events.content.ContentEvent;
import com.atlassian.confluence.event.events.content.blogpost.BlogPostEvent;
import com.atlassian.confluence.event.events.content.comment.CommentEvent;
import com.atlassian.confluence.event.events.content.page.PageEvent;
import com.atlassian.confluence.event.events.space.SpaceEvent;
import com.atlassian.confluence.event.events.types.Created;
import com.atlassian.confluence.event.events.types.Removed;
import com.atlassian.confluence.event.events.types.Updated;
import com.atlassian.confluence.event.events.types.Viewed;
import com.atlassian.confluence.ext.usage.UsageConstants;
import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.Comment;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.Spaced;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.core.task.Task;
import com.atlassian.event.Event;
import org.apache.lucene.document.DateTools;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;

import java.util.Date;

/**
 * Index events as the come through.
 */
public class UsageEventWrapperTask implements Task {

    private Date dateFired;
    private Document document;
    private String eventType;
    private String entityType;
    private long entityId;
    private long spaceId;

    public UsageEventWrapperTask(Event event, Extractor extractor) {
        this.dateFired = new Date();
        initDocument(event, extractor);
    }

    public void execute() throws Exception {
        throw new UnsupportedOperationException("Task shouldn't be executed directly");
    }

    public Date getDateFired() {
        return dateFired;
    }

    public String getEventType() {
        return eventType;
    }

    public String getEntityType() {
        return entityType;
    }

    public long getEntityId() {
        return entityId;
    }

    public long getSpaceId() {
        return spaceId;
    }

    public String toString() {
        return "UsageEventWrapperTask: " + eventType + ":" + entityType + "(" + entityId + ") at " + dateFired;
    }

    public Document getDocument() {
        return document;
    }

    /**
     * We create the document here so that objects aren't kept around in memory while the task is queued.
     *
     * @param e         event
     * @param extractor extractor
     */
    public void initDocument(Event e, Extractor extractor) {
        document = new Document();

        document.add(new Field(UsageConstants.FIELD_DATE, DateTools.dateToString(getDateFired(), DateTools.Resolution.SECOND), Field.Store.YES, Field.Index.NOT_ANALYZED));

        if (e instanceof Viewed)
            eventType = UsageConstants.TYPE_VIEW;
        else if (e instanceof Updated)
            eventType = UsageConstants.TYPE_UPDATE;
        else if (e instanceof Created)
            eventType = UsageConstants.TYPE_CREATE;
        else if (e instanceof Removed)
            eventType = UsageConstants.TYPE_REMOVE;

        if (eventType != null)
            document.add(new Field(UsageConstants.FIELD_EVENTTYPE, eventType, Field.Store.YES, Field.Index.NOT_ANALYZED));

        Space eventSpace = null;
        if (e instanceof PageEvent) {
            entityType = Page.CONTENT_TYPE;
            eventSpace = ((PageEvent) e).getPage().getSpace();
        } else if (e instanceof SpaceEvent) {
            entityId = ((SpaceEvent) e).getSpace().getId();
            entityType = UsageConstants.SPACE_ENTITY_TYPE;
            eventSpace = ((SpaceEvent) e).getSpace();
        } else if (e instanceof CommentEvent) {
            entityType = Comment.CONTENT_TYPE;
            final ContentEntityObject container = ((CommentEvent) e).getComment().getContainer();
            if (container instanceof Spaced) {
                eventSpace = ((Spaced) container).getSpace();
            }
        } else if (e instanceof BlogPostEvent) {
            entityType = BlogPost.CONTENT_TYPE;
            eventSpace = ((BlogPostEvent) e).getBlogPost().getSpace();
        }

        if (entityType != null) {
            document.add(new Field(UsageConstants.FIELD_ENTITYTYPE, entityType, Field.Store.YES, Field.Index.NOT_ANALYZED));
        }

        if (e instanceof ContentEvent) {
            entityId = ((ContentEvent) e).getContent().getId();
        }

        document.add(new Field(UsageConstants.FIELD_ENTITYID, Long.toString(entityId), Field.Store.YES, Field.Index.NOT_ANALYZED));

        // check for null space reference (CONF-8747)
        // NPE thrown when viewing an older revision (no space is attached to old revisions)
        if (eventSpace != null) {
            spaceId = eventSpace.getId();
            document.add(new Field(UsageConstants.FIELD_SPACEID, Long.toString(spaceId), Field.Store.YES, Field.Index.NOT_ANALYZED));
        }

        String user = AuthenticatedUserThreadLocal.getUsername();

        if (user != null) {
            document.add(new Field("user", user, Field.Store.YES, Field.Index.NOT_ANALYZED));
        }

        if (e instanceof ContentEvent) {
            ContentEvent contentEvent = (ContentEvent) e;
            if (extractor != null) {
                extractor.addFields(document, null, contentEvent.getContent());
            }
        }
    }
}
