package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.confluence.extra.flyingpdf.PdfExportProgressMonitor;
import com.atlassian.confluence.extra.flyingpdf.util.ErrorMessages;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.core.util.ProgressMeter;

/**
 * An implementation of the <code>PdfExportProgressMonitor</code> which translates the calls it receives into the
 * Confluence <code>ProgressMeter</code> which is used during long running tasks.
 * <p>
 * Used only when documents are converted via the Sandbox
 */
public class SandboxProgressMonitor implements PdfExportProgressMonitor {
    /**
     * The percentage of the total work that is typically spent calculating the content tree. This is completely
     * dependent on the amount of content selected for export so this is a bit of a joke.
     */
    private static final int CONTENT_TREE_CALCULATION_PERCENTAGE = 4;

    /**
     * The percentage of the total work that is typically spent performing wiki to HTML conversion.
     */
    private static final int HTML_CONVERSION_PERCENTAGE = 4;

    /**
     * i18n the status messages
     */
    private final I18NBean i18NBean;

    /**
     * Helper for error messages
     */
    private final ErrorMessages errorMessages;

    /**
     * The underlying progress meter to be updated
     */
    private final ProgressMeter progressMeter;

    /**
     * The total number of pages that will be exported - updated during the operation.
     */
    private int numberOfPages;

    /**
     * An ongoing count of the number of pages that have been converted.
     */
    private int convertedPageCount = 0;

    public SandboxProgressMonitor(I18NBean bean, ErrorMessages errorMessages, ProgressMeter progressMeter) {
        i18NBean = bean;
        this.errorMessages = errorMessages;
        this.progressMeter = progressMeter;
    }

    public void started() {
        progressMeter.setPercentage(0);
        progressMeter.setStatus(i18NBean.getText("com.atlassian.confluence.extra.flyingpdf.progress.started"));
    }

    public void beginCalculationOfContentTree() {
        progressMeter.setStatus(i18NBean
                .getText("com.atlassian.confluence.extra.flyingpdf.progress.calculatecontenttree"));
    }

    @Override
    public void completedCalculationOfPdfPages(int numberOfPages) {
        this.numberOfPages = numberOfPages;
        this.convertedPageCount = 0;
    }

    public void completedCalculationOfContentTree(int numberOfPages) {
        this.numberOfPages = numberOfPages;
        progressMeter.setStatus(i18NBean
                .getText("com.atlassian.confluence.extra.flyingpdf.progress.contenttreecalculated"));
        progressMeter.setPercentage(CONTENT_TREE_CALCULATION_PERCENTAGE);
    }

    private static double getPercentage(double progress, double total) {
        return progress * 100 / total;
    }

    private static double scalePercentage(double originalPercentage, double scalingFactor) {
        return scalingFactor / 100 * originalPercentage;
    }

    public void completedExportedHtmlConversionForPage(String pageId, String pageTitle) {
        convertedPageCount++;

        double percentageComplete = getPercentage(convertedPageCount, numberOfPages);
        double weightedPercentageComplete = scalePercentage(percentageComplete, HTML_CONVERSION_PERCENTAGE)
                + CONTENT_TREE_CALCULATION_PERCENTAGE;
        progressMeter.setPercentage((int) weightedPercentageComplete);

        String message = i18NBean.getText("com.atlassian.confluence.extra.flyingpdf.progress.pageConvertedToHtml",
                new String[]{String.valueOf(convertedPageCount), String.valueOf(numberOfPages)});
        progressMeter.setStatus(message);
    }

    public void beginHtmlToPdfConversion() {
        convertedPageCount = 0; // reset to zero - have counted up for HTMl conversion
        progressMeter.setPercentage(CONTENT_TREE_CALCULATION_PERCENTAGE + HTML_CONVERSION_PERCENTAGE);
        progressMeter.setStatus(i18NBean.getText("com.atlassian.confluence.extra.flyingpdf.progress.beginhtmltopdf"));
    }

    public void performingHtmlToPdfConversionForPage(String pageTitle) {
        convertedPageCount++;

        double percentageComplete = getPercentage(convertedPageCount, numberOfPages);
        int pdfConversionPercentage = 100 - CONTENT_TREE_CALCULATION_PERCENTAGE - HTML_CONVERSION_PERCENTAGE;
        double weightedPercentageComplete = scalePercentage(percentageComplete, pdfConversionPercentage)
                + CONTENT_TREE_CALCULATION_PERCENTAGE + HTML_CONVERSION_PERCENTAGE;

        int calculatedPercentage = (int) weightedPercentageComplete;
        // ensure double calculations don't end up rounding to 100% since a percentage value of 100 is the mechanism
        // used by the long running task framework to decide on completion.
        // bugs in pdf conversion progress tracking may also mean that the number of pages converted is more than the predicted total!
        if (calculatedPercentage > 99)
            calculatedPercentage = 99;

        progressMeter.setPercentage(calculatedPercentage);

        String message = i18NBean.getText("com.atlassian.confluence.extra.flyingpdf.progress.pageConvertedToPdf",
                new String[]{String.valueOf(convertedPageCount), String.valueOf(numberOfPages)});
        progressMeter.setStatus(message);
    }

    public void completed(String downloadPath) {
        progressMeter.setCompletedSuccessfully(true);
        progressMeter.setStatus(i18NBean.getText("com.atlassian.confluence.extra.flyingpdf.progress.completed",
                new String[]{downloadPath}));
        progressMeter.setPercentage(100);
    }

    public void errored(String exceptionMessage) {
        progressMeter.setCompletedSuccessfully(false);
        progressMeter.setStatus(exceptionMessage);
    }
}
