package com.atlassian.confluence.extra.flyingpdf.html;

import com.atlassian.confluence.content.render.xhtml.DefaultConversionContext;
import com.atlassian.confluence.content.render.xhtml.Renderer;
import com.atlassian.confluence.extra.flyingpdf.PdfExportProgressMonitor;
import com.atlassian.confluence.extra.flyingpdf.config.PdfExportSettingsManager;
import com.atlassian.confluence.extra.flyingpdf.impl.NoOpProgressMonitor;
import com.atlassian.confluence.extra.flyingpdf.impl.PdfResourceManager;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.importexport.impl.ExportFileNameGenerator;
import com.atlassian.confluence.pages.AbstractPage;
import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.ContentNode;
import com.atlassian.confluence.pages.ContentTree;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.renderer.PageContext;
import com.atlassian.confluence.setup.bandana.ConfluenceBandanaContext;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.confluence.util.velocity.VelocityUtils;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.renderer.RenderContext;
import com.google.common.collect.ImmutableList;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.apache.velocity.VelocityContext;
import org.springframework.stereotype.Component;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import static com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy.DecorationComponent.FOOTER;
import static com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy.DecorationComponent.HEADER;
import static com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy.DecorationComponent.PAGE_NUMBERS;
import static com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy.DecorationComponent.TITLE_PAGE;
import static com.atlassian.confluence.extra.flyingpdf.html.LinkFixer.InternalPageStrategy.ANCHOR;
import static com.atlassian.confluence.extra.flyingpdf.html.LinkFixer.InternalPageStrategy.NORMALISE;
@Component
public class RenderedXhtmlBuilder implements XhtmlBuilder {

    static final String MAIN_STYLE_ID = "confluence.flyingpdf.styleId";

    /**
     * Indicates how many characters can be fit into one line. If the length of content exceeds this value,
     * then we adjust the CSS for the table so that it can be printed out without cut-off.
     */
    private static final int CHARACTER_PER_LINE = Integer.getInteger("confluence.flyingpdf.default.characters.per.line", 80);
    private static final Pattern HEADING_PATTERN = Pattern.compile("\\</?h(\\d)(\\>|\\s)");
    private static final String PAGE_TEMPLATE_NAME = "/templates/extra/pdfexport/pagehtml.vm";
    private static final String COMPLETE_EXPORT_PAGE_TEMPLATE_NAME = "/templates/extra/pdfexport/completeexport.vm";
    private static final String TOC_TEMPLATE_NAME = "/templates/extra/pdfexport/toc.vm";
    private static final String CONFLUENCE_BASE_STYLES = loadResource("master.css");
    private static final Logger LOG = Logger.getLogger(RenderedXhtmlBuilder.class);

    private final I18NBeanFactory i18NBeanFactory;
    private final Renderer xhtmlRenderer;
    private final SettingsManager settingsManager;
    private final ExportHtmlService exportHtmlService;
    private final PdfExportSettingsManager pdfSettings;
    private final PdfResourceManager pdfResourceManager;
    private final ExportFileNameGenerator htmlExportFileNameGenerator;

    public RenderedXhtmlBuilder(
            @ComponentImport I18NBeanFactory i18NBeanFactory,
            @ComponentImport Renderer xhtmlRenderer,
            @ComponentImport SettingsManager settingsManager,
            ExportHtmlService exportHtmlService,
            PdfExportSettingsManager pdfSettings,
            PdfResourceManager pdfResourceManager,
            ExportFileNameGenerator htmlExportFileNameGenerator) {
        this.i18NBeanFactory = i18NBeanFactory;
        this.xhtmlRenderer = xhtmlRenderer;
        this.settingsManager = settingsManager;
        this.exportHtmlService = exportHtmlService;
        this.pdfSettings = pdfSettings;
        this.pdfResourceManager = pdfResourceManager;
        this.htmlExportFileNameGenerator = htmlExportFileNameGenerator;
    }

    @Override
    public Document buildHtml(ContentTree contentTree, Space space,
                              LinkRenderingDetails linkRendering, DecorationPolicy decoration) throws ImportExportException {
        // call with a progress monitor that doesn't actually care e.g. perhaps only a single quick page being exported.
        return buildHtml(contentTree, space, linkRendering, decoration, new NoOpProgressMonitor());
    }

    @Override
    public Document buildHtml(ContentTree contentTree, Space space, LinkRenderingDetails linkRendering,
                              DecorationPolicy decoration, PdfExportProgressMonitor progress)
            throws ImportExportException {
        TocBuilder tocBuilder = new TocBuilder();
        BookmarksBuilder bookmarksBuilder = new BookmarksBuilder();

        List<String> pageHtml = renderContentTreeNodes(contentTree.getRootNodes(), tocBuilder, bookmarksBuilder, 0,
                contentTree, progress);

        LinkFixer linkFixer = new LinkFixer(space.getKey(),
                settingsManager.getGlobalSettings().getBaseUrl(), linkRendering.getLinkStrategy());

        populateLinkFixer(linkFixer, contentTree, linkRendering.getInternalPages());

        return buildHtml(pageHtml, space, decoration, tocBuilder, bookmarksBuilder, linkFixer);
    }

    @Override
    public Document buildHtml(BlogPost blogPost) throws ImportExportException {
        List<String> pageHtml = ImmutableList.of(renderToHtml(blogPost, null));

        LinkFixer linkFixer = new LinkFixer(blogPost.getSpace().getKey(),
                settingsManager.getGlobalSettings().getBaseUrl(), ANCHOR);

        return buildHtml(pageHtml, blogPost.getSpace(),
                DecorationPolicy.none(), new TocBuilder(), new BookmarksBuilder(), linkFixer);
    }

    private Document buildHtml(List<String> pageHtml, Space space, DecorationPolicy decoration, TocBuilder tocBuilder,
                               BookmarksBuilder bookmarksBuilder, LinkFixer linkFixer) throws ImportExportException {
        Reader htmlReader = createCompleteExportHtml(pageHtml, tocBuilder, space, decoration);

        // convert the HTML 4 to an XHTML page
        try {
            HtmlToDomParser domParser = HtmlConverterUtils.getHtmlToXhtmlParser(linkFixer);
            Document xhtmlDocument = domParser.parse(htmlReader);
            addTableLayout(xhtmlDocument);

            insertBookmarkElement(xhtmlDocument, bookmarksBuilder);

            AutoFontScaleUtils.applyTableScalingLogic(xhtmlDocument);

            return xhtmlDocument;
        } finally {
            try {
                htmlReader.close();
            } catch (IOException ex) {
                LOG.warn("Exception while closing the intermediate HTML file for reading.");
            }
        }
    }

    /**
     * Adjusts the CSS for table if exists.
     * For more info see CONF-34390
     *
     * @param xhtmlDocument - the document object to be adjusted based on the width of headers.
     */
    private void addTableLayout(final Document xhtmlDocument) {
        //add the class - fixedTableLayout - to a table if the content of table headers is too long to be fit into a page.
        final NodeList tables = xhtmlDocument.getElementsByTagName("table");
        if (tables.getLength() == 0) {
            return;
        }
        for (int tableIndex = 0; tableIndex < tables.getLength(); tableIndex++) {
            final Element table = (Element) tables.item(tableIndex);
            fixTableLayout(table);
        }
    }

    /**
     * Applies the fixed table layout by checking the length of header first, then the body.
     * Unfortunately, we have to check all rows here as it is possible a row down the bottom
     * could have contents being cut-off.
     *
     * @param table - represents the table element in DOM.
     */
    private void fixTableLayout(final Element table) {
        final NodeList headRow = getTableHeader(table);
        if (headRow != null && rowIsTooLong(headRow)) {
            fixTableStyle(table);
            return;
        }
        fixTableByBody(table);
    }

    private void fixTableByBody(final Element table) {
        final Element body = getFirstElementByTagName(table, "tbody");
        if (null != body) {
            final NodeList children = body.getElementsByTagName("tr");
            //iterates the rows, and stops at first long row to save the time.
            for (int index = 0; index < children.getLength(); index++) {
                final Node row = children.item(index);
                if (rowIsTooLong(row.getChildNodes())) {
                    fixTableStyle(table);
                    return;
                }
            }
        }
    }

    private Element getFirstElementByTagName(final Element element, final String tag) {
        final NodeList elements = element.getElementsByTagName(tag);
        if (elements.getLength() > 0) {
            return (Element) elements.item(0);
        }
        return null;
    }

    private void fixTableStyle(final Element table) {
        //see the corresponding CSS in master.css
        table.setAttribute("class", table.getAttribute("class") + " fixedTableLayout");
        final Element colGroup = getFirstElementByTagName(table, "colgroup");
        if (null != colGroup) {
            final NodeList cols = colGroup.getElementsByTagName("col");
            for (int index = 0; index < cols.getLength(); index++) {
                final Element col = (Element) cols.item(index);
                //remove the hard coded width from the <col> element if exists.
                col.removeAttribute("style");
            }
        }
    }

    private boolean rowIsTooLong(final NodeList row) {
        int characterCount = 0;
        for (int col = 0; col < row.getLength(); col++) {
            characterCount += getColLength(row.item(col));
        }
        return characterCount > CHARACTER_PER_LINE;
    }

    private int getColLength(final Node col) {
        final String content = col.getTextContent();
        return content == null ? 0 : content.length();
    }

    private NodeList getTableHeader(final Element table) {
        final Element header = getFirstElementByTagName(table, "thead");
        if (null != header) {
            Element row = getFirstElementByTagName(header, "tr");
            return row == null ? null : row.getChildNodes();
        }
        return null;
    }

    private void populateLinkFixer(LinkFixer linkFixer, ContentTree contentTree, Collection<Page> additionalInternalPages) {
        List<ContentNode> contentNodes = contentTree.getAllContentNodes();
        for (ContentNode node : contentNodes) {
            Page p = node.getPage();
            linkFixer.addPage(p.getIdAsString(), p.getTitle());
        }
        additionalInternalPages.forEach(p -> linkFixer.addPage(p.getIdAsString(), p.getTitle()));
    }

    @Override
    public Document generateTableOfContents(String baseUrl, Space space, TocBuilder tocBuilder) throws ImportExportException {
        VelocityContext context = createCompleteVelocityContext(Collections.emptyList(), tocBuilder, space,
                DecorationPolicy.titlePage());

        StringWriter writer = new StringWriter();
        try {
            exportHtmlService.renderTemplateWithoutSwallowingErrors(TOC_TEMPLATE_NAME, context, writer);
        } catch (Exception ex) {
            throw new ImportExportException("Failure while rendering the " + TOC_TEMPLATE_NAME, ex);
        }

        HtmlToDomParser domParser = HtmlConverterUtils.getHtmlToXhtmlParser(new LinkFixer(space.getKey(), baseUrl, NORMALISE));
        Document xhtmlDocument = domParser.parse(new StringReader((writer).getBuffer().toString()));
        addTableLayout(xhtmlDocument);

        final BookmarksBuilder bookmarks = new BookmarksBuilder();
        bookmarks.beginEntry(i18NBeanFactory.getI18NBean().getText("com.atlassian.confluence.extra.flyingpdf.toc"));
        bookmarks.endEntry();
        insertBookmarkElement(xhtmlDocument, bookmarks);

        AutoFontScaleUtils.applyTableScalingLogic(xhtmlDocument);
        return xhtmlDocument;
    }

    /**
     * Create the complete HTML for export and return a Reader suitable for reading this data. Depending on the number
     * of pages rendered the complete HTML will either be created in memory or on the file system. e.g. if only a single
     * page is being exported then it will be stored in memory.
     *
     * @return a Reader suitable for reading the complete HTML. The client is responsible for closing this Reader when
     * they are finished.
     */
    private Reader createCompleteExportHtml(List<String> renderedPages, TocBuilder tocBuilder,
                                            Space space, DecorationPolicy decoration) throws ImportExportException {
        RenderOutput output;
        if (renderedPages.size() > 1) {
            output = new FileRenderOutput(htmlExportFileNameGenerator, "export", "intermediate");
        } else {
            output = new StringRenderOutput();
        }
        VelocityContext context = createCompleteVelocityContext(renderedPages, tocBuilder, space, decoration);
        Writer writer;
        try {
            writer = output.getOutputWriter();
        } catch (IOException ex) {
            throw new ImportExportException("Failed to open output writer for the intermediate HTML file.", ex);
        }
        try {
            exportHtmlService.renderTemplateWithoutSwallowingErrors(COMPLETE_EXPORT_PAGE_TEMPLATE_NAME, context, writer);
        } catch (Exception ex) {
            throw new ImportExportException("Failure while rendering the " + COMPLETE_EXPORT_PAGE_TEMPLATE_NAME, ex);
        } finally {
            try {
                writer.close();
            } catch (IOException ex) {
                LOG.warn("Failed to close the intermediate HTML file during PDF export.", ex);
            }
        }
        try {
            return output.getResultReader();
        } catch (IOException ex) {
            throw new ImportExportException("Failed to open the intermediate HTML file for reading.");
        }
    }

    private VelocityContext createCompleteVelocityContext(List<String> renderedPages, TocBuilder tocBuilder,
                                                          Space currentSpace, DecorationPolicy decoration) {
        Map<String, Object> contextMap = new HashMap<>(8);

        if (decoration.components().contains(HEADER)) {
            String header = getHeader(currentSpace);
            if (!StringUtils.isEmpty(header))
                contextMap.put("headerHtml", header);
        }
        if (decoration.components().contains(FOOTER)) {
            String footer = getFooter(currentSpace);
            if (!StringUtils.isEmpty(footer))
                contextMap.put("footerHtml", footer);
        }
        if (decoration.components().contains(TITLE_PAGE)) {
            String titlePage = getTitlePage(currentSpace);
            if (!StringUtils.isEmpty(titlePage))
                contextMap.put("titleHtml", titlePage);
        }
        if (decoration.components().contains(PAGE_NUMBERS)){
            contextMap.put("pageNumbers", true);
        }
        String customStyles = getUserStyles(currentSpace);
        String userStyle = CONFLUENCE_BASE_STYLES;
        if (customStyles != null)
            userStyle += customStyles;
        if (!StringUtils.isEmpty(userStyle))
            contextMap.put("userStyleHtml", userStyle);

        contextMap.put("styleId", MAIN_STYLE_ID);
        contextMap.put("tocEntries", tocBuilder.getEntries());
        contextMap.put("pdfResourceManager", pdfResourceManager);
        contextMap.put("pages", renderedPages);
        return new VelocityContext(contextMap);
    }

    private String getUserStyles(Space currentSpace) {
        // try to get space settings then fallback to global
        String customStyles = pdfSettings.getStyle(new ConfluenceBandanaContext(currentSpace));
        if (StringUtils.isEmpty(customStyles)) {
            customStyles = pdfSettings.getStyle(new ConfluenceBandanaContext());
        }
        if (StringUtils.isNotEmpty(customStyles))
            return customStyles;
        else
            return "";
    }

    private String getTitlePage(Space currentSpace) {
        // try to get space settings then fallback to global
        String titlePage = pdfSettings.getTitlePage(new ConfluenceBandanaContext(currentSpace));
        if (StringUtils.isEmpty(titlePage)) {
            titlePage = pdfSettings.getTitlePage(new ConfluenceBandanaContext());
        }
        return titlePage;
    }

    private String getFooter(Space currentSpace) {
        // try to get space settings then fallback to global
        String footer = pdfSettings.getFooter(new ConfluenceBandanaContext(currentSpace));
        if (StringUtils.isEmpty(footer)) {
            footer = pdfSettings.getFooter(new ConfluenceBandanaContext());
        }
        return footer;
    }

    private String getHeader(Space currentSpace) {
        // try to get space settings then fallback to global
        String header = pdfSettings.getHeader(new ConfluenceBandanaContext(currentSpace));
        if (StringUtils.isEmpty(header)) {
            header = pdfSettings.getHeader(new ConfluenceBandanaContext());
        }
        return header;
    }

    /**
     * Render each of the nodes (Pages) in the supplied List to an HTML formatted String.
     *
     * @param nodes            The list of nodes to be rendered to HTML
     * @param tocBuilder       the class to be informed as nodes relevant to the table of contents are processed
     * @param bookmarksBuilder the class to be informed as nodes relevant to the bookmarks are processed
     * @param level            the level in the ContentTree currently being processed.
     * @param fullContentTree  the full content tree from which the nodes being rendered are taken.
     * @param progress         a class interested in progress updates about the process
     * @return a list of the Strings where each String is an HTML version of the page content for each note.
     */
    private List<String> renderContentTreeNodes(List<ContentNode> nodes, TocBuilder tocBuilder, BookmarksBuilder bookmarksBuilder,
                                                int level, ContentTree fullContentTree, PdfExportProgressMonitor progress) {
        List<String> renderedPagesContent = new ArrayList<>();
        for (ContentNode node : nodes) {
            Page page = node.getPage();
            String renderedHtml = renderToHtml(page, fullContentTree);
            renderedPagesContent.add(renderedHtml);
            progress.completedExportedHtmlConversionForPage(String.valueOf(page.getId()), page.getTitle());

            tocBuilder.addEntry(level, page.getTitle());
            bookmarksBuilder.beginEntry(page.getTitle());
            List<ContentNode> children = node.getChildren();
            if (children != null && !children.isEmpty()) {
                renderedPagesContent.addAll(renderContentTreeNodes(children, tocBuilder, bookmarksBuilder, level + 1,
                        fullContentTree, progress));
            }
            bookmarksBuilder.endEntry();
        }
        return renderedPagesContent;
    }

    private String renderToHtml(AbstractPage page, ContentTree fullContentTree) {
        if (LOG.isDebugEnabled())
            LOG.debug("Rendering to exported XHTML page id=" + page.getId() + " (" + page.getTitle() + ")");

        PageContext context = page.toPageContext();
        context.setBaseUrl(settingsManager.getGlobalSettings().getBaseUrl());
        context.setOutputType(RenderContext.PDF);

        DefaultConversionContext conversionContext = new DefaultConversionContext(context);
        if (fullContentTree != null) {
            conversionContext.setContentTree(fullContentTree);
        }
        final String html = xhtmlRenderer.render(page, conversionContext);
        return renderPageTemplate(page.getTitle(), html);
    }

    /**
     * Render the individual page Velocity template using the supplied variables and return the full result as a String.
     *
     * @param title     the title of the page being rendered
     * @param content   the HTML 4 content of the page
     * @return the rendered output for the page.
     */
    private String renderPageTemplate(String title, String content) {
        // Create Velocity Context for the individual page template
        Map<String, Object> contextMap = new HashMap<>(2);
        contextMap.put("pageTitle", title);
        contextMap.put("contentHtml", content);

        return VelocityUtils.getRenderedTemplate(PAGE_TEMPLATE_NAME, contextMap);
    }

    private void insertBookmarkElement(Document document, BookmarksBuilder builder) {
        List<BookmarksBuilder.BookmarkEntry> topLevelBookmarks = builder.getEntries();
        if (topLevelBookmarks.isEmpty())
            return;

        Element bookmarksElement = document.createElement("bookmarks");

        NodeList headList = document.getElementsByTagName("head");
        if (headList.getLength() < 1)
            return;

        Node headNode = headList.item(0);
        headNode.appendChild(bookmarksElement);

        appendBookmarksElement(document, bookmarksElement, topLevelBookmarks);
    }

    /**
     * @param document        the document that newly created elements will belong to
     * @param parentNode      the node that the elements will be appended to
     * @param bookmarkEntries the list of entries to create bookmark nodes for
     */
    private void appendBookmarksElement(Document document, Node parentNode, List<BookmarksBuilder.BookmarkEntry> bookmarkEntries) {
        for (BookmarksBuilder.BookmarkEntry entry : bookmarkEntries) {
            Element bookmarkElement = document.createElement("bookmark");
            bookmarkElement.setAttribute("name", entry.getTitle());
            bookmarkElement.setAttribute("href", "#" + entry.getTitle());
            parentNode.appendChild(bookmarkElement);

            if (entry.hasChildEntries())
                appendBookmarksElement(document, bookmarkElement, entry.getChildEntries());
        }
    }

    private static String loadResource(String masterCss) {
        try {
            InputStream in = RenderedXhtmlBuilder.class.getResourceAsStream("/templates/extra/pdfexport/" + masterCss);
            String ret = IOUtils.toString(in, "ASCII");
            IOUtils.closeQuietly(in);
            return ret;
        } catch (Throwable t) { // don't want this to bork the loading of the plugin so catch everything
            if (LOG != null) {
                LOG.error("Unable to load the default styles for PDF export", t);
            }
            return "";
        }
    }

    private interface RenderOutput {
        Writer getOutputWriter() throws IOException;
        Reader getResultReader() throws IOException;
    }

    private static class FileRenderOutput implements RenderOutput {
        private final File outputFile;

        FileRenderOutput(ExportFileNameGenerator fileNameGenerator, String... distinguishers)
                throws ImportExportException {
            try {
                this.outputFile = fileNameGenerator.getExportFile(distinguishers);
            } catch (IOException ex) {
                throw new ImportExportException("Failed to create output file during PDF export.");
            }
        }

        public Writer getOutputWriter() throws IOException {
            return new BufferedWriter(new OutputStreamWriter(new FileOutputStream(outputFile), StandardCharsets.UTF_8));
        }

        public Reader getResultReader() throws IOException {
            return new BufferedReader(new InputStreamReader(new FileInputStream(outputFile), StandardCharsets.UTF_8));
        }
    }

    private static class StringRenderOutput implements RenderOutput {
        private final StringWriter writer = new StringWriter();

        public Writer getOutputWriter() {
            return writer;
        }

        public Reader getResultReader() {
            return new StringReader(writer.getBuffer().toString());
        }

    }

}