package com.atlassian.confluence.extra.flyingpdf.html;

import org.apache.commons.lang3.StringUtils;
import org.apache.xerces.xni.Augmentations;
import org.apache.xerces.xni.QName;
import org.apache.xerces.xni.XMLAttributes;
import org.apache.xerces.xni.XMLString;
import org.apache.xerces.xni.XNIException;
import org.apache.xerces.xni.parser.XMLComponentManager;
import org.apache.xerces.xni.parser.XMLConfigurationException;
import org.cyberneko.html.filters.DefaultFilter;

import java.util.Stack;

public class ConfluenceHtmlToXmlFilter
        extends DefaultFilter {
    private Stack<QName> styleStack = new Stack<>();
    private StringBuffer collectedStyles;
    private LinkFixer linkFixer;
    private boolean insideStyle = false;

    @Override
    public void reset(XMLComponentManager componentManager)
            throws XMLConfigurationException {
        collectedStyles = (StringBuffer) componentManager.getProperty(HtmlConverterUtils.STYLECOLLECTOR_KEY);
        linkFixer = (LinkFixer) componentManager.getProperty(HtmlConverterUtils.LINKFIXER_KEY);
        super.reset(componentManager);
    }

    public void emptyElement(QName element, XMLAttributes attributes,
                             Augmentations augs) throws XNIException {
        element.localpart = element.localpart == null ? null : element.localpart.toLowerCase();
        element.prefix = element.prefix == null ? null : element.prefix.toLowerCase();
        element.uri = element.uri == null ? null : element.uri.toLowerCase();
        element.rawname = element.rawname == null ? null : element.rawname.toLowerCase();

        if ("img".equals(element.localpart)) {
            String attrName = "border";
            String formatStr = "border-style:solid;border-width:%1$s;";
            moveAttributeToStyle(attributes, attrName, formatStr);

            moveAttributeToStyle(attributes, "height", "height:%1$s;");
            moveAttributeToStyle(attributes, "width", "width:%1$s;");
        }

        super.emptyElement(element, attributes, augs);
    }

    public void endElement(QName element, Augmentations augs)
            throws XNIException {
        element.localpart = element.localpart == null ? null : element.localpart.toLowerCase();
        element.prefix = element.prefix == null ? null : element.prefix.toLowerCase();
        element.uri = element.uri == null ? null : element.uri.toLowerCase();
        element.rawname = element.rawname == null ? null : element.rawname.toLowerCase();

        if ("style".equals(element.localpart)) {
            if (!styleStack.isEmpty())
                styleStack.pop();

            insideStyle = !styleStack.isEmpty();
        }
        super.endElement(element, augs);
    }

    public void startElement(QName element, XMLAttributes attributes,
                             Augmentations augs) throws XNIException {
        element.localpart = element.localpart == null ? null : element.localpart.toLowerCase();
        element.prefix = element.prefix == null ? null : element.prefix.toLowerCase();
        element.uri = element.uri == null ? null : element.uri.toLowerCase();
        element.rawname = element.rawname == null ? null : element.rawname.toLowerCase();

        // Handle the font tag,
        if ("font".equals(element.localpart)) {
            String attrName = "color";
            moveAttributeToStyle(attributes, attrName, "color:%1$s;");
        } else if (element.localpart.equals("style")) {
            String id = attributes.getValue("id");
            if (id == null || !id.equals(RenderedXhtmlBuilder.MAIN_STYLE_ID)) {
                styleStack.push(element);
                insideStyle = true;
            }
        } else if (element.localpart.equals("a")) {
            int hrefIdx = attributes.getIndex("href");
            if (hrefIdx != -1) {
                String href = attributes.getValue(hrefIdx);
                if (!StringUtils.isBlank(href)) {
                    String newHref = linkFixer.convertLink(href);
                    if (newHref != null) {
                        attributes.setValue(hrefIdx, newHref);
                    }
                }
            }
        }
        super.startElement(element, attributes, augs);
    }

    private void moveAttributeToStyle(XMLAttributes attributes,
                                      String attrName, String formatStr) {
        String attrValue = attributes.getValue(attrName);
        if (!StringUtils.isEmpty(attrValue)) {
            String newStyle = String.format(formatStr, attrValue);
            int styleIdx = attributes.getIndex("style");
            if (styleIdx != -1) {
                String style = attributes.getValue(styleIdx);
                newStyle = newStyle + style;
                attributes.setValue(styleIdx, newStyle);
            } else {
                attributes.addAttribute(new QName(null, "style", "style", null), "CDATA", newStyle);
            }
        }
    }


    public void characters(XMLString text, Augmentations augs)
            throws XNIException {
        if (insideStyle) {
            collectedStyles.append(text.toString()).append("\r\n");
        } else {
            super.characters(text, augs);
        }
    }

    public StringBuffer getCollectedStyles() {
        return collectedStyles;
    }
}
