package com.atlassian.confluence.extra.flyingpdf.auding;

import com.atlassian.audit.api.AuditService;
import com.atlassian.audit.entity.AuditEvent;
import com.atlassian.audit.entity.AuditResource;
import com.atlassian.audit.entity.AuditType;
import com.atlassian.audit.entity.CoverageArea;
import com.atlassian.audit.entity.CoverageLevel;
import com.atlassian.confluence.audit.StandardAuditResourceTypes;
import com.atlassian.confluence.extra.flyingpdf.impl.PdfExportEvent;
import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.Page;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Arrays;
import java.util.List;

import static com.atlassian.confluence.extra.flyingpdf.analytic.ExportScope.PAGE;
import static com.atlassian.confluence.extra.flyingpdf.analytic.ExportScope.SPACE;

@Component
public class PdfExportAuditListener {

    private static final Logger log = LoggerFactory.getLogger(PdfExportAuditListener.class);

    public static final String AUDIT_CATEGORY_IMPORT_EXPORT = "audit.logging.category.import.export";

    public static final String AUDIT_SUMMARY_SPACE_KEY = "audit.logging.summary.space.pdf.export";
    public static final String AUDIT_SUMMARY_PAGE_KEY = "audit.logging.summary.page.pdf.export";
    public static final String AUDIT_SUMMARY_BLOG_KEY = "audit.logging.summary.blog.pdf.export";

    private final EventPublisher eventPublisher;
    private final AuditService auditService;

    private final StandardAuditResourceTypes resourceTypes;

    public PdfExportAuditListener(@ComponentImport EventPublisher eventPublisher,
        @ComponentImport AuditService auditService,
        @ComponentImport StandardAuditResourceTypes resourceTypes) {

        this.eventPublisher = eventPublisher;
        this.auditService = auditService;
        this.resourceTypes = resourceTypes;
    }

    @PostConstruct
    public void init() {
        eventPublisher.register(this);
    }

    @PreDestroy
    public void destroy() {
        eventPublisher.unregister(this);
    }

    @EventListener
    public void onPDFExport(PdfExportEvent event) {

        if (event.getExportScope() == SPACE) {
            log.info("Auditing PDF export of space");
            performSpaceAudit(event);
            return;
        }

        if (event.getExportScope() == PAGE && Page.CONTENT_TYPE.equals(event.getPageType())) {
            log.info("Auditing PDF export of page");
            performPageAudit(event);
            return;
        }

        if (event.getExportScope() == PAGE && BlogPost.CONTENT_TYPE.equals(event.getPageType())) {
            log.info("Auditing PDF export of blog post");
            performBlogAudit(event);
            return;
        }

        log.warn("Unknown PDF export type for scope '{}' and page type '{}'. No audit is performed on this action",
                event.getExportScope(), event.getPageType());
    }

    private void performSpaceAudit(PdfExportEvent event) {
        AuditResource affectedSpace = AuditResource.builder(event.getSpaceName(), resourceTypes.space())
                .id(String.valueOf(event.getSpaceId()))
                .build();

        performAudit(AUDIT_CATEGORY_IMPORT_EXPORT, AUDIT_SUMMARY_SPACE_KEY, CoverageArea.LOCAL_CONFIG_AND_ADMINISTRATION,
                affectedSpace);
    }

    private void performPageAudit(PdfExportEvent event) {
        AuditResource affectedSpace = AuditResource.builder(event.getSpaceName(), resourceTypes.space())
                .id(String.valueOf(event.getSpaceId()))
                .build();

        AuditResource affectedPage = AuditResource.builder(event.getPageTitle(), resourceTypes.page())
                .id(String.valueOf(event.getPageId()))
                .build();

        performAudit(AUDIT_CATEGORY_IMPORT_EXPORT, AUDIT_SUMMARY_PAGE_KEY, CoverageArea.END_USER_ACTIVITY,
                affectedSpace, affectedPage);
    }

    private void performBlogAudit(PdfExportEvent event) {
        AuditResource affectedSpace = AuditResource.builder(event.getSpaceName(), resourceTypes.space())
                .id(String.valueOf(event.getSpaceId()))
                .build();

        AuditResource affectedBlog = AuditResource.builder(event.getPageTitle(), resourceTypes.page())
                .id(String.valueOf(event.getPageId()))
                .build();

        performAudit(AUDIT_CATEGORY_IMPORT_EXPORT, AUDIT_SUMMARY_BLOG_KEY, CoverageArea.END_USER_ACTIVITY,
                affectedSpace, affectedBlog);
    }

    private void performAudit(String categoryKey, String summaryKey, CoverageArea area, AuditResource ... affectedObjects) {
        List<AuditResource> affectedObjectsList = Arrays.asList(affectedObjects);
        AuditType auditType = AuditType.fromI18nKeys(area, CoverageLevel.ADVANCED, categoryKey, summaryKey).build();
        auditService.audit(AuditEvent.builder(auditType).appendAffectedObjects(affectedObjectsList).build());
        affectedObjectsList.forEach(auditResource ->
                log.info("Audit - affected object [{}] id: {}", auditResource.getType(), auditResource.getId()));
    }
}
