package com.atlassian.confluence.extra.flyingpdf.util;

import com.atlassian.annotations.Internal;
import com.google.common.collect.ImmutableList;
import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.pdf.PdfCopy;
import com.lowagie.text.pdf.PdfImportedPage;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.SimpleBookmark;

import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Merges partial PDF files together and converts link between them to bookmarks
 */
@Internal
public class PdfJoiner {

    private List<Map> finalOutlines = new ArrayList<>();

    private ExportedSpaceStructure structure;

    private String outputFile;

    public PdfJoiner(String outputFile, ExportedSpaceStructure structure) {
        this.structure = structure;
        this.outputFile = outputFile;
    }

    public void join() throws IOException, DocumentException {
        // Here we DFS our forest of pages and join them in single ordered list
        final List<PdfNode> forest = ImmutableList.<PdfNode>builder()
                .add(structure.getTableOfContents())
                .addAll(structure.getConfluencePages())
                .build();

        mergePdf(forest, outputFile);
    }

    private void mergePdf(List<PdfNode> forest, String outputFile) throws DocumentException, IOException {
        final Document document = new Document();
        final PdfCopy writer = new PdfCopy(document, new FileOutputStream(outputFile));
        document.open();


        for (PdfNode node : forest) {
            // Process each page in order
            process(writer, node, null);
        }

        // Outlines is a thing that can be displayed in the left panel of every PDF viewer. This is like
        // implicit table of contents in each PDF document. Outlines are filled during PDF join
        if (!finalOutlines.isEmpty()) {
            writer.setOutlines(finalOutlines);
        }

        writer.flush();
        document.close();
    }

    /**
     * Appends an input pdf file into output and returns its bookmarks.
     */
    private void process(PdfCopy writer, PdfNode node, List<Map<String, List>> parentOutlines) throws IOException, DocumentException {
        final PdfReader reader = new PdfReader(node.getFilename());

        int numberOfPages = reader.getNumberOfPages();

        // Here we shift outlines for the given page based on starting page location
        final List<Map<String, List>> currentOutlines = shiftOutlinesAddress(node, parentOutlines, reader);

        // Write pages to output file
        for (int i = 1; i <= numberOfPages; i++) {
            PdfImportedPage page = writer.getImportedPage(reader, i);
            writer.addPage(page);
        }

        writer.freeReader(reader);
        reader.close();

        for (PdfNode child : node.getChildren()) {
            process(writer, child, currentOutlines);
        }
    }

    /**
     * Shift the outlines for current page based on its location
     */
    private List<Map<String, List>> shiftOutlinesAddress(PdfNode node, List<Map<String, List>> parentBookmarks, PdfReader reader) {
        //noinspection unchecked
        List<Map<String, List>> currentBookmarks = SimpleBookmark.getBookmark(reader);
        if (currentBookmarks == null) {
            currentBookmarks = new ArrayList<>();
        }

        if (currentBookmarks.isEmpty()) {
            currentBookmarks.add(new HashMap<>());
        }

        final int location = structure.locationByNode(node);
        if (location > 1) {
            SimpleBookmark.shiftPageNumbers(currentBookmarks, location - 1, null);
        }

        // top level bookmark
        if (parentBookmarks == null) {
            finalOutlines.addAll(currentBookmarks);
        } else {
            Map<String, List> first = parentBookmarks.get(0);
            //noinspection unchecked
            first.computeIfAbsent("Kids", k -> new ArrayList<>()).add(currentBookmarks.get(0));
        }
        return currentBookmarks;
    }

}
